<?php
// Set the content type to JSON
header('Content-Type: application/json');

// Debug log file
$debug_log = 'hubtel_callback_debug_log.txt';
function log_debug($msg) {
    global $debug_log;
    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - $msg\n", FILE_APPEND);
}

// Get raw POST data and decode JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);
log_debug("Received callback: " . $input);

// Simulate storage for transaction status
$transaction_status_file = 'transaction_status.json';
if (!file_exists($transaction_status_file)) file_put_contents($transaction_status_file, '{}');
$transactions = json_decode(file_get_contents($transaction_status_file), true);

// Extract transaction details from callback
$transaction_id = $data['Data']['ClientReference'] ?? null;
$status = strtolower($data['Message'] ?? 'pending'); // will be 'success', 'failed', etc.
$message = $data['Message'] ?? null;

// Log the callback for debugging purposes
log_debug("Parsed: Transaction ID: $transaction_id, Status: $status, Message: $message");

// Check if this is a callback notification
if ($transaction_id && $message) {
    // Update the transaction status in the JSON file
    $transactions[$transaction_id] = [
        'status' => $status,
        'message' => $message
    ];
    file_put_contents($transaction_status_file, json_encode($transactions));
    log_debug("Updated transaction_status.json for $transaction_id");

    // Update the transaction_history table if status is success or failed
    if ($status === 'success' || $status === 'failed') {
        require_once __DIR__ . '/dashboard/database.php';
        session_start();
        $conn = getDBConnection();
        $new_status = ($status === 'success') ? 'Paid' : 'Failed';
        $stmt = $conn->prepare("UPDATE transaction_history SET status=? WHERE transaction_id=?");
        $stmt->bind_param('ss', $new_status, $transaction_id);
        $stmt->execute();
        $stmt->close();
        log_debug("Updated transaction_history status to $new_status for $transaction_id");

        // If SUCCESS, proceed with user/account logic
        if ($status === 'success') {
            // 1. Select all details for this transaction
            $stmt = $conn->prepare("SELECT * FROM transaction_history WHERE transaction_id=? LIMIT 1");
            $stmt->bind_param('s', $transaction_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $transaction = $result->fetch_assoc();
            $stmt->close();
            log_debug("Fetched transaction_history row: " . json_encode($transaction));

            if ($transaction) {
                $_SESSION['transaction'] = $transaction;
                $username = $transaction['username'];

                // 2. Try to get password from hotspot_users
                $stmt = $conn->prepare("SELECT password FROM hotspot_users WHERE username=? LIMIT 1");
                $stmt->bind_param('s', $username);
                $stmt->execute();
                $stmt->bind_result($user_password);
                $found_in_users = $stmt->fetch();
                $stmt->close();
                log_debug("Checked hotspot_users for $username: found=" . ($found_in_users ? 'yes' : 'no'));

                // 3. If not found, try hotspot_accounts
                $account_password = null;
                $found_in_accounts = false;
                if (!$found_in_users) {
                    $stmt = $conn->prepare("SELECT password FROM hotspot_accounts WHERE username=? LIMIT 1");
                    $stmt->bind_param('s', $username);
                    $stmt->execute();
                    $stmt->bind_result($account_password);
                    $found_in_accounts = $stmt->fetch();
                    $stmt->close();
                    log_debug("Checked hotspot_accounts for $username: found=" . ($found_in_accounts ? 'yes' : 'no'));
                }

                // 4. Store password in session
                if ($found_in_users) {
                    $_SESSION['transaction']['password'] = $user_password;
                    log_debug("Password from hotspot_users stored in session.");
                } elseif (!empty($account_password)) {
                    $_SESSION['transaction']['password'] = $account_password;
                    log_debug("Password from hotspot_accounts stored in session.");
                } else {
                    // Generate random 8-digit password
                    $random_password = strval(rand(10000000, 99999999));
                    $_SESSION['transaction']['password'] = $random_password;
                    log_debug("Generated random password: $random_password");
                }

                // Prepare values, converting empty or nulls to PHP null and using correct types
                $profile = $transaction['profile'] ?? null;
                $limit_uptime = $transaction['limit_uptime'] ?? null;
                $limit_bytes_in = (isset($transaction['limit_bytes_in']) && $transaction['limit_bytes_in'] !== '' && $transaction['limit_bytes_in'] !== null) ? (int)$transaction['limit_bytes_in'] : null;
                $limit_bytes_out = (isset($transaction['limit_bytes_out']) && $transaction['limit_bytes_out'] !== '' && $transaction['limit_bytes_out'] !== null) ? (int)$transaction['limit_bytes_out'] : null;
                $limit_bytes_total = (isset($transaction['limit_bytes_total']) && $transaction['limit_bytes_total'] !== '' && $transaction['limit_bytes_total'] !== null) ? (int)$transaction['limit_bytes_total'] : null;

                log_debug("Preparing to upsert hotspot_accounts with values: username=$username, password={$_SESSION['transaction']['password']}, account_name=$username, profile=$profile, limit_uptime=$limit_uptime, limit_bytes_in=" . var_export($limit_bytes_in, true) . ", limit_bytes_out=" . var_export($limit_bytes_out, true) . ", limit_bytes_total=" . var_export($limit_bytes_total, true));

                // Instead of upsert logic, send POST request to upsert_hotspot_account.php
                $account_data = [
                    'username' => $username,
                    'password' => $_SESSION['transaction']['password'],
                    'account_name' => $username,
                    'enabled' => '1',
                    'profile' => $profile,
                    'limit_uptime' => $limit_uptime,
                    'limit_bytes_in' => $limit_bytes_in,
                    'limit_bytes_out' => $limit_bytes_out,
                    'limit_bytes_total' => $limit_bytes_total,
                    'transaction_id' => $transaction_id
                ];

                $ch = curl_init('https://ewusiwahostel.com/admin/upsert_hotspot_account.php');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($account_data));
                curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
                $response = curl_exec($ch);
                curl_close($ch);

                log_debug("Upsert hotspot_accounts response: $response");

                // 7. Select all details from hotspot_accounts for this username
                $stmt = $conn->prepare("SELECT * FROM hotspot_accounts WHERE username=? LIMIT 1");
                $stmt->bind_param('s', $username);
                $stmt->execute();
                $account_result = $stmt->get_result();
                $account = $account_result->fetch_assoc();
                $stmt->close();
                log_debug("Fetched hotspot_accounts row: " . json_encode($account));

                if ($account) {
                    // 8. Generate MikroTik commands
                    $remove_cmd = "/ip hotspot user remove [find name=\"{$account['username']}\"]";
                    $add_cmd = "/ip hotspot user add name=\"{$account['username']}\" password=\"{$account['password']}\" profile=\"{$account['profile']}\"";
                    if (!empty($account['limit_uptime'])) {
                        $add_cmd .= " limit-uptime=\"{$account['limit_uptime']}\"";
                    }
                    if (!empty($account['limit_bytes_in'])) {
                        $add_cmd .= " limit-bytes-in={$account['limit_bytes_in']}";
                    }
                    if (!empty($account['limit_bytes_out'])) {
                        $add_cmd .= " limit-bytes-out={$account['limit_bytes_out']}";
                    }
                    if (!empty($account['limit_bytes_total'])) {
                        $add_cmd .= " limit-bytes-total={$account['limit_bytes_total']}";
                    }
                    log_debug("Generated MikroTik commands:\nREMOVE: $remove_cmd\nADD: $add_cmd");

                    // 9. Insert commands into mikrotik_commands table
                    $router_id = 'pending';
                    $now = date('Y-m-d H:i:s');
                    $stmt = $conn->prepare("INSERT INTO mikrotik_commands (command, router_id, created_at) VALUES (?, ?, ?), (?, ?, ?)");
                    $stmt->bind_param(
                        'ssssss',
                        $remove_cmd, $router_id, $now,
                        $add_cmd, $router_id, $now
                    );
                    $stmt->execute();
                    $affected = $stmt->affected_rows;
                    $stmt->close();
                    log_debug("Inserted MikroTik commands into mikrotik_commands. Rows affected: $affected");
                } else {
                    log_debug("No account found in hotspot_accounts for $username. No MikroTik commands generated.");
                }
            } else {
                log_debug("No transaction found in transaction_history for $transaction_id.");
            }
        }
    }

    // --- Name lookup and update account_name in transaction_history ---
    $stmt = $conn->prepare("SELECT momo_number FROM transaction_history WHERE transaction_id=? LIMIT 1");
    $stmt->bind_param('s', $transaction_id);
    $stmt->execute();
    $stmt->bind_result($momo_number);
    $found = $stmt->fetch();
    $stmt->close();

    if ($found && !empty($momo_number)) {
        $lookup_url = "https://bundle.krolatech.org/korba_lookup.php";
        $lookup_payload = json_encode(['mobile_number' => $momo_number]);
        $ch = curl_init($lookup_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $lookup_payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        $lookup_response = curl_exec($ch);
        $lookup_http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        log_debug("Name lookup response ($lookup_http_code): $lookup_response");

        if ($lookup_http_code == 200 && $lookup_response) {
            $lookup_data = json_decode($lookup_response, true);
            if (is_array($lookup_data) && isset($lookup_data['account_name'])) {
                $account_name_from_lookup = $lookup_data['account_name'];

                $update_stmt = $conn->prepare("UPDATE transaction_history SET account_name=? WHERE momo_number=?");
                $update_stmt->bind_param('ss', $account_name_from_lookup, $momo_number);
                $update_stmt->execute();
                $update_stmt->close();

                log_debug("Updated account_name for momo_number $momo_number to $account_name_from_lookup in transaction_history");
            } else {
                log_debug("Lookup response missing expected fields.");
            }
        } else {
            log_debug("Failed to get valid response from lookup endpoint.");
        }
    } else {
        log_debug("No momo_number found for transaction_id $transaction_id.");
    }

    echo json_encode(['success' => true]);
    if (isset($conn)) $conn->close();
} else if ($transaction_id) {
    // If only transaction_id is provided, return the current status
    if (isset($transactions[$transaction_id])) {
        echo json_encode($transactions[$transaction_id]);
    } else {
        echo json_encode([
            'status' => 'PENDING',
            'message' => 'Transaction is still pending approval.'
        ]);
    }
} else {
    echo json_encode([
        'error' => 'Invalid request parameters.'
    ]);
}
?>