<?php
// --- Logging functions ---
function log_error($message) {
    $logFile = __DIR__ . '/hubtel_error.log';
    $entry = "[" . date('Y-m-d H:i:s') . "] ERROR: $message\n";
    file_put_contents($logFile, $entry, FILE_APPEND);
}

function log_response($response) {
    $logFile = __DIR__ . '/hubtel_response.log';
    $entry = "[" . date('Y-m-d H:i:s') . "] RESPONSE: $response\n";
    file_put_contents($logFile, $entry, FILE_APPEND);
}

function log_step($step, $info = '') {
    $logFile = __DIR__ . '/hubtel_steps.log';
    $entry = "[" . date('Y-m-d H:i:s') . "] STEP: $step | $info\n";
    file_put_contents($logFile, $entry, FILE_APPEND);
}

function log_received_data($data) {
    $logFile = __DIR__ . '/hubtel_received_data.log';
    $entry = "[" . date('Y-m-d H:i:s') . "] DATA: " . json_encode($data) . "\n";
    file_put_contents($logFile, $entry, FILE_APPEND);
}

// Accept JSON input
header('Content-Type: application/json');
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Log received data for debugging
log_step('Received request', $input);
log_received_data($data);

try {
    log_step('Validating required fields');
    $requiredFields = ['customer_number', 'amount', 'transaction_id', 'network_code', 'callback_url', 'description'];
    foreach ($requiredFields as $field) {
        if (empty($data[$field])) {
            log_error("Missing required field: " . str_replace('_', ' ', $field));
            log_step('Validation failed', "Missing field: $field");
            throw new Exception("Missing required field: " . str_replace('_', ' ', $field));
        }
    }

    log_step('Formatting phone number');
    $phone = $data['customer_number'];
    if (substr($phone, 0, 3) !== '233') {
        $phone = '233' . ltrim($phone, '0');
    }

    log_step('Preparing API request data');
    $requestData = [
        'CustomerMsisdn' => $phone,
        'Channel' => $data['network_code'],
        'Amount' => floatval($data['amount']),
        'PrimaryCallbackUrl' => $data['callback_url'],
        'Description' => $data['description'],
        'ClientReference' => $data['transaction_id']
    ];
    log_step('Request data', json_encode($requestData));

    // Hubtel API credentials (REPLACE WITH YOUR ACTUAL VALUES)
    $posSalesId = '2030712';
    $username = '8Ojxny3';
    $password = 'fcdff8e680ea44aa868fc57fa22da25c';

    log_step('Sending request to Hubtel API');
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://rmp.hubtel.com/merchantaccount/merchants/$posSalesId/receive/mobilemoney");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
    curl_setopt($ch, CURLOPT_USERPWD, "$username:$password");
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json',
        'Cache-Control: no-cache'
    ]);
    // curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // For local testing only

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    log_response($response);
    log_step('Received response from Hubtel', "HTTP Code: $httpCode");

    if (curl_errno($ch)) {
        log_error('CURL Error: ' . curl_error($ch));
        log_step('CURL Error', curl_error($ch));
        throw new Exception('CURL Error: ' . curl_error($ch));
    }
    curl_close($ch);

    $result = json_decode($response, true);

    if ($httpCode !== 200) {
        $errorMsg = $result['Message'] ?? 'API request failed';
        log_error("Hubtel Error: $errorMsg (HTTP $httpCode)");
        log_step('Hubtel Error', $errorMsg);
        throw new Exception("Hubtel Error: $errorMsg (HTTP $httpCode)");
    }

    if ($result['ResponseCode'] !== '0001') {
        log_error("Transaction failed: " . $result['Message']);
        log_step('Transaction failed', $result['Message']);
        throw new Exception("Transaction failed: " . $result['Message']);
    }

    $transactionId = $result['Data']['TransactionId'];
    log_step('Transaction successful', "Transaction ID: $transactionId");

    echo json_encode([
        'status' => 'success',
        'transaction_id' => $transactionId,
        'message' => $result['Message']
    ]);
    exit;

} catch (Exception $e) {
    log_error($e->getMessage());
    log_step('Exception', $e->getMessage());
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
    exit;
}