<?php
session_start();
header('Content-Type: application/json');

// Basic security checks
if (!isset($_SESSION['phone_number'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

require_once 'database.php';

$data = json_decode(file_get_contents('php://input'), true);

// --- Input Validation ---
$message = $data['sms_message'] ?? '';
$template_name = $data['template_name'] ?? '';
$audience = $data['target_audience'] ?? 'ALL';

if (empty($message)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'SMS message cannot be empty.']);
    exit;
}

$conn = getDBConnection();
$response_data = ['success' => true, 'message' => ''];

try {
    // --- 1. Save new template if a name is provided ---
    if (!empty($template_name)) {
        $stmt = $conn->prepare("INSERT INTO sms_templates (template_name, message_body) VALUES (?, ?)");
        $stmt->bind_param('ss', $template_name, $message);
        $stmt->execute();
        $stmt->close();
        $response_data['message'] .= "Template '{$template_name}' saved successfully. ";
    }

    // --- 2. Fetch recipient phone numbers ---
    if ($audience === 'ALL') {
        $sql = "SELECT DISTINCT momo_number FROM transaction_history WHERE momo_number IS NOT NULL AND momo_number != ''";
        $stmt = $conn->prepare($sql);
    } else {
        $sql = "SELECT DISTINCT momo_number FROM transaction_history WHERE router = ? AND momo_number IS NOT NULL AND momo_number != ''";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("s", $audience);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $recipients = [];
    while ($row = $result->fetch_assoc()) {
        $recipients[] = $row['momo_number'];
    }
    $stmt->close();

    if (empty($recipients)) {
        $response_data['message'] .= 'No recipients found for the selected audience.';
        echo json_encode($response_data);
        exit;
    }

    // --- 3. Send SMS to each recipient using Hubtel ---
    $sent_count = 0;
    // Prepare the statement once before the loop for efficiency
    $log_stmt = $conn->prepare("INSERT INTO sent_sms_logs (recipient, message_content, response_code, rate, message_id, status, status_description, network_id, raw_response, sent_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

    foreach ($recipients as $recipient) {
        $fields = [
            'From' => 'PlutoWiFi',
            'To' => $recipient,
            'Content' => $message,
            'ClientReference' => 'campaign_' . time() . '_' . $recipient
        ];

        $headers = [
            "Authorization: Basic " . base64_encode("poxywkdp:demvuqjr"), // Note: Consider moving credentials to a config file
            "Content-Type: application/json"
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://sms.hubtel.com/v1/messages/send");
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $sms_response = curl_exec($ch);
        $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        // Log each SMS attempt to a text file
        file_put_contents('campaign_sms_log.txt', date('Y-m-d H:i:s') . " - To: $recipient, Code: $responseCode, Response: $sms_response\n", FILE_APPEND);

        if ($responseCode >= 200 && $responseCode < 300) {
            $sent_count++;
        }

        // --- Log each sent message to the database ---
        $responseData = json_decode($sms_response, true);
        $rate = $responseData['rate'] ?? null;
        $message_id = $responseData['messageId'] ?? null;
        $status = $responseData['status'] ?? null;
        $status_description = $responseData['statusDescription'] ?? null;
        $network_id = $responseData['networkId'] ?? null;

        $log_stmt->bind_param(
            'ssidsisss',
            $recipient,
            $message,
            $responseCode,
            $rate,
            $message_id,
            $status,
            $status_description,
            $network_id,
            $sms_response
        );
        $log_stmt->execute();
    }

    $response_data['message'] .= "Attempted to send SMS to {$sent_count} out of " . count($recipients) . " recipients.";

} catch (mysqli_sql_exception $e) {
    http_response_code(500);
    // Check for specific error for missing table
    if (strpos($e->getMessage(), "Table 'emceqagn_pluto.sms_templates' doesn't exist") !== false) {
        $response_data = ['success' => false, 'message' => "Error: The 'sms_templates' table does not exist in the database. Please create it to save templates."];
    } else {
        $response_data = ['success' => false, 'message' => 'A database error occurred: ' . $e->getMessage()];
    }
} catch (Exception $e) {
    http_response_code(500);
    $response_data = ['success' => false, 'message' => 'An unexpected error occurred: ' . $e->getMessage()];
}

echo json_encode($response_data);
?>