<?php
session_start();
define('SESSION_TIMEOUT', 900); // 15 minutes.

// Handle logout
if (isset($_GET['logout'])) {
    session_unset();
    session_destroy();
    header("Location: auth-boxed-signin.php");
    exit();
}

if (!isset($_SESSION['phone_number'])) {
    header("Location: auth-boxed-signin.php");
    exit();
}
if (isset($_SESSION['LAST_ACTIVITY']) && (time() - $_SESSION['LAST_ACTIVITY'] > SESSION_TIMEOUT)) {
    session_unset();
    session_destroy();
    header("Location: auth-boxed-signin.php?timeout=1");
    exit();
}
$_SESSION['LAST_ACTIVITY'] = time();

require_once 'database.php';
$conn = getDBConnection();

$searchUser = trim($_GET['search_user'] ?? '');
$userWhere = '';
$userParams = [];
if ($searchUser !== '') {
    $userWhere = " AND (username = ? OR momo_number = ?) ";
    $userParams = [$searchUser, $searchUser];
}

// Helper to get revenue for a date range
function getRevenue($conn, $start, $end, $userWhere = '', $userParams = []) {
    $sql = "SELECT SUM(amount) as total FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere";
    $stmt = $conn->prepare($sql);
    if ($userWhere) {
        $stmt->bind_param("ssss", $start, $end, $userParams[0], $userParams[1]);
    } else {
        $stmt->bind_param("ss", $start, $end);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return floatval($result['total'] ?? 0);
}

// Helper to get daily revenue for a week

function getDailyRevenue($conn, $start, $userWhere = '', $userParams = []) {
    $data = [];
    for ($i = 0; $i < 7; $i++) {
        $day = date('Y-m-d', strtotime("$start +$i day"));
        $next = date('Y-m-d', strtotime("$day +1 day"));
        $data[] = getRevenue($conn, $day, $next, $userWhere, $userParams);
    }
    return $data;
}

function getWeeklyRevenue($conn, $monthStart, $userWhere = '', $userParams = []) {
    $data = [];
    for ($i = 0; $i < 5; $i++) {
        $weekStart = date('Y-m-d', strtotime("$monthStart +$i week"));
        $weekEnd = date('Y-m-d', strtotime("$weekStart +7 days"));
        $data[] = getRevenue($conn, $weekStart, $weekEnd, $userWhere, $userParams);
    }
    return $data;
}

function getMonthlyRevenue($conn, $yearStart, $userWhere = '', $userParams = []) {
    $data = [];
    for ($i = 0; $i < 12; $i++) {
        $monthStart = date('Y-m-01', strtotime("$yearStart +$i month"));
        $monthEnd = date('Y-m-01', strtotime("$monthStart +1 month"));
        $data[] = getRevenue($conn, $monthStart, $monthEnd, $userWhere, $userParams);
    }
    return $data;
}

// Date ranges
$thisWeekStart = date('Y-m-d', strtotime('monday this week'));
$lastWeekStart = date('Y-m-d', strtotime('monday last week'));
$lastWeekEnd = date('Y-m-d', strtotime('sunday last week +1 day'));

$thisMonthStart = date('Y-m-01');
$lastMonthStart = date('Y-m-01', strtotime('first day of last month'));
$lastMonthEnd = date('Y-m-01');

$thisYearStart = date('Y-01-01');
$lastYearStart = date('Y-01-01', strtotime('last year'));
$lastYearEnd = date('Y-01-01');

// Revenue breakdowns
$revenue_this_week = getDailyRevenue($conn, $thisWeekStart, $userWhere, $userParams);
$revenue_last_week = getDailyRevenue($conn, $lastWeekStart, $userWhere, $userParams);

$revenue_this_month = getWeeklyRevenue($conn, $thisMonthStart, $userWhere, $userParams);
$revenue_last_month = getWeeklyRevenue($conn, $lastMonthStart, $userWhere, $userParams);

$revenue_this_year = getMonthlyRevenue($conn, $thisYearStart, $userWhere, $userParams);
$revenue_last_year = getMonthlyRevenue($conn, $lastYearStart, $userWhere, $userParams);

// Totals
$total_this_week = array_sum($revenue_this_week);
$total_last_week = array_sum($revenue_last_week);

$total_this_month = array_sum($revenue_this_month);
$total_last_month = array_sum($revenue_last_month);

$total_this_year = array_sum($revenue_this_year);
$total_last_year = array_sum($revenue_last_year);

// Get all profiles
$profiles = [];
$sql = "SELECT DISTINCT profile FROM transaction_history WHERE profile IS NOT NULL AND profile != ''";
if ($userWhere) {
    $sql .= $userWhere;
}
$res = $conn->prepare($sql);
if ($userWhere) {
    $param1 = $userParams[0];
    $param2 = $userParams[1];
    $res->bind_param("ss", $param1, $param2);
}
$res->execute();
$result = $res->get_result();
while ($row = $result->fetch_assoc()) {
    $profiles[] = $row['profile'];
}


// Calculate total revenue for each profile for this week
$color_palette = [
    '#1b55e2', '#e7515a', '#00ab55', '#e2a03f', '#3cba92', '#f09819', '#4361ee', '#b224ef', '#7579ff', '#ff5858'
];
// $profile_revenues = [];
// $profile_colors = [];
// foreach ($profiles as $i => $profile) {
//     $stmt = $conn->prepare("SELECT SUM(amount) as total FROM transaction_history WHERE profile = ? AND created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere");
//     if ($userWhere) {
//         $stmt->bind_param("ssssss", $profile, $thisWeekStart, date('Y-m-d', strtotime($thisWeekStart . ' +7 days')), $userParams[0], $userParams[1]);
//     } else {
//         $stmt->bind_param("sss", $profile, $thisWeekStart, date('Y-m-d', strtotime($thisWeekStart . ' +7 days')));
//     }
//     $stmt->execute();
//     $result = $stmt->get_result()->fetch_assoc();
//     $profile_revenues[] = floatval($result['total'] ?? 0);
//     $profile_colors[] = $color_palette[$i % count($color_palette)];
    
// }
$profile_revenues = [];
$profile_colors = [];
foreach ($profiles as $i => $profile) {
    $sql = "SELECT SUM(amount) as total FROM transaction_history WHERE profile = ? AND created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere";
    $stmt = $conn->prepare($sql);
    $weekStart = $thisWeekStart;
    $weekEnd = date('Y-m-d', strtotime($thisWeekStart . ' +7 days'));
    if ($userWhere) {
        $param1 = $profile;
        $param2 = $weekStart;
        $param3 = $weekEnd;
        $param4 = $userParams[0];
        $param5 = $userParams[1];
        $stmt->bind_param("sssss", $param1, $param2, $param3, $param4, $param5);
    } else {
        $param1 = $profile;
        $param2 = $weekStart;
        $param3 = $weekEnd;
        $stmt->bind_param("sss", $param1, $param2, $param3);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $profile_revenues[] = floatval($result['total'] ?? 0);
    $profile_colors[] = $color_palette[$i % count($color_palette)];
    
}
// $total_profile_revenue = array_sum($profile_revenues);

// Get today's date range
$today = date('Y-m-d');
$tomorrow = date('Y-m-d', strtotime('+1 day'));

// Get top 2 selling profiles for today (by count)
$topProfilesToday = [];
$sql = "SELECT profile, COUNT(*) as cnt FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere GROUP BY profile ORDER BY cnt DESC LIMIT 2";
$stmt = $conn->prepare($sql);
if ($userWhere) {
    $stmt->bind_param("ssss", $today, $tomorrow, $userParams[0], $userParams[1]);
} else {
    $stmt->bind_param("ss", $today, $tomorrow);
}
$stmt->execute();
$res = $stmt->get_result();
while ($row = $res->fetch_assoc()) {
    $topProfilesToday[] = $row;
}


// Example: Total sales for today
$stmt2 = $conn->prepare("SELECT COUNT(*) as total FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere");
if ($userWhere) {
    $stmt2->bind_param("ssss", $today, $tomorrow, $userParams[0], $userParams[1]);
} else {
    $stmt2->bind_param("ss", $today, $tomorrow);
}
$stmt2->execute();
$totalToday = $stmt2->get_result()->fetch_assoc()['total'] ?? 0;

// Get total sales for today
$stmt2 = $conn->prepare("SELECT COUNT(*) as total FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid'");
$stmt2->bind_param("ss", $today, $tomorrow);
$stmt2->execute();
$totalToday = $stmt2->get_result()->fetch_assoc()['total'] ?? 0;

// Prepare donut chart data
$donutLabels = [];
$donutCounts = [];
$donutColors = ['#1b55e2', '#e7515a', '#00ab55'];
$topCount = 0;
foreach ($topProfilesToday as $i => $row) {
    $donutLabels[] = $row['profile'];
    $donutCounts[] = intval($row['cnt']);
    $topCount += intval($row['cnt']);
}
$others = $totalToday - $topCount;
if ($others > 0) {
    $donutLabels[] = 'Others';
    $donutCounts[] = $others;
}

// Helper to get daily sales count for a week
function getDailySalesCount($conn, $start, $userWhere = '', $userParams = []) {
    $data = [];
    for ($i = 0; $i < 7; $i++) {
        $day = date('Y-m-d', strtotime("$start +$i day"));
        $next = date('Y-m-d', strtotime("$day +1 day"));
        $sql = "SELECT COUNT(*) as cnt FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere";
        $stmt = $conn->prepare($sql);
        if ($userWhere) {
            $param1 = $day;
            $param2 = $next;
            $param3 = $userParams[0];
            $param4 = $userParams[1];
            $stmt->bind_param("ssss", $param1, $param2, $param3, $param4);
        } else {
            $param1 = $day;
            $param2 = $next;
            $stmt->bind_param("ss", $param1, $param2);
        }
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $data[] = intval($result['cnt'] ?? 0);
    }
    return $data;
}


$daily_sales_this_week = getDailySalesCount($conn, $thisWeekStart, $userWhere, $userParams);
$daily_sales_last_week = getDailySalesCount($conn, $lastWeekStart, $userWhere, $userParams);

// Helper to get total orders for a date range
function getOrders($conn, $start, $end, $userWhere = '', $userParams = []) {
    $sql = "SELECT COUNT(*) as total FROM transaction_history WHERE created_at >= ? AND created_at < ? AND status = 'Paid' $userWhere";
    $stmt = $conn->prepare($sql);
    if ($userWhere) {
        $param1 = $start;
        $param2 = $end;
        $param3 = $userParams[0];
        $param4 = $userParams[1];
        $stmt->bind_param("ssss", $param1, $param2, $param3, $param4);
    } else {
        $param1 = $start;
        $param2 = $end;
        $stmt->bind_param("ss", $param1, $param2);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    return intval($result['total'] ?? 0);
}

function getWeeklyOrders($conn, $monthStart, $userWhere = '', $userParams = []) {
    $data = [];
    for ($i = 0; $i < 5; $i++) {
        $weekStart = date('Y-m-d', strtotime("$monthStart +$i week"));
        $weekEnd = date('Y-m-d', strtotime("$weekStart +7 days"));
        $data[] = getOrders($conn, $weekStart, $weekEnd, $userWhere, $userParams);
    }
    return $data;
}


$total_orders_this_month = getOrders($conn, $thisMonthStart, date('Y-m-d', strtotime("$thisMonthStart +1 month")), $userWhere, $userParams);
$weekly_orders_this_month = getWeeklyOrders($conn, $thisMonthStart, $userWhere, $userParams);

// Get top 5 profiles and their sale stats (total sold, unit amount, total amount), descending by total sold
$profileStats = [];
$sql = "
    SELECT 
        profile, 
        COUNT(*) as sold, 
        AVG(amount) as unit_amount, 
        SUM(amount) as total_amount,
        MIN(source) as source
    FROM transaction_history
    WHERE status = 'Paid' $userWhere
    GROUP BY profile
    ORDER BY sold DESC
    LIMIT 5
";
$stmt = $conn->prepare($sql);
if ($userWhere) {
    $param1 = $userParams[0];
    $param2 = $userParams[1];
    $stmt->bind_param("ss", $param1, $param2);
}
$stmt->execute();
$res = $stmt->get_result();
$profileStats = [];
while ($row = $res->fetch_assoc()) {
    $profileStats[] = $row;
}$sql = "
    SELECT 
        profile, 
        COUNT(*) as sold, 
        AVG(amount) as unit_amount, 
        SUM(amount) as total_amount,
        MIN(source) as source
    FROM transaction_history
    WHERE status = 'Paid' $userWhere
    GROUP BY profile
    ORDER BY sold DESC
    LIMIT 5
";
$stmt = $conn->prepare($sql);
if ($userWhere) {
    $param1 = $userParams[0];
    $param2 = $userParams[1];
    $stmt->bind_param("ss", $param1, $param2);
}
$stmt->execute();
$res = $stmt->get_result();
$profileStats = [];
while ($row = $res->fetch_assoc()) {
    $profileStats[] = $row;
}
// Total Successful Transactions (Paid)
$sqlPaid = "SELECT SUM(amount) as total FROM transaction_history WHERE status = 'Paid' $userWhere";
$stmtPaid = $conn->prepare($sqlPaid);
if ($userWhere) {
    $param1 = $userParams[0];
    $param2 = $userParams[1];
    $stmtPaid->bind_param("ss", $param1, $param2);
}
$stmtPaid->execute();
$totalPaid = $stmtPaid->get_result()->fetch_assoc()['total'] ?? 0;


// Total Failed Transactions (Failed)
$sqlFailed = "SELECT SUM(amount) as total FROM transaction_history WHERE status = 'Failed' $userWhere";
$stmtFailed = $conn->prepare($sqlFailed);
if ($userWhere) {
    $param1 = $userParams[0];
    $param2 = $userParams[1];
    $stmtFailed->bind_param("ss", $param1, $param2);
}
$stmtFailed->execute();
$totalFailed = $stmtFailed->get_result()->fetch_assoc()['total'] ?? 0;


// Total Attempted Transactions (Paid + Failed)
$totalAttempted = floatval($totalPaid) + floatval($totalFailed);

?>

<!DOCTYPE html>
<html lang="en" dir="ltr">
    <head>
        <meta charset="utf-8">
        <meta http-equiv="X-UA-Compatible" content="IE=edge">
        <title>Pluto Wifi</title>
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <link rel="icon" type="image/x-icon" href="favicon.png">
        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin="">
        <link href="css2?family=Nunito:wght@400;500;600;700;800&display=swap" rel="stylesheet">
        <link rel="stylesheet" type="text/css" media="screen" href="assets/css/perfect-scrollbar.min.css">
        <link rel="stylesheet" type="text/css" media="screen" href="assets/css/style.css">
        <link defer="" rel="stylesheet" type="text/css" media="screen" href="assets/css/animate.css">
        <script src="assets/js/perfect-scrollbar.min.js"></script>
        <script defer="" src="assets/js/popper.min.js"></script>
        <script defer="" src="assets/js/tippy-bundle.umd.min.js"></script>
        <script defer="" src="assets/js/sweetalert.min.js"></script>
    </head>

    <body x-data="main" class="relative overflow-x-hidden font-nunito text-sm font-normal antialiased" :class="[ $store.app.sidebar ? 'toggle-sidebar' : '', $store.app.theme === 'dark' || $store.app.isDarkMode ?  'dark' : '', $store.app.menu, $store.app.layout,$store.app.rtlClass]">
        <!-- sidebar menu overlay -->
        <div x-cloak="" class="fixed inset-0 z-50 bg-[black]/60 lg:hidden" :class="{'hidden' : !$store.app.sidebar}" @click="$store.app.toggleSidebar()"></div>

        <!-- screen loader -->
        <div class="screen_loader animate__animated fixed inset-0 z-[60] grid place-content-center bg-[#fafafa] dark:bg-[#060818]">
            <img src="logo.jpg" alt="Loading..." width="96" height="96" class="animate-pulse rounded-full shadow-lg" />
        </div>

        <!-- scroll to top button -->
        <div class="fixed bottom-6 z-50 ltr:right-6 rtl:left-6" x-data="scrollToTop">
            <template x-if="showTopButton">
                <button type="button" class="btn btn-outline-primary animate-pulse rounded-full bg-[#fafafa] p-2 dark:bg-[#060818] dark:hover:bg-primary" @click="goToTop">
                    <svg width="24" height="24" class="h-4 w-4" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path opacity="0.5" fill-rule="evenodd" clip-rule="evenodd" d="M12 20.75C12.4142 20.75 12.75 20.4142 12.75 20L12.75 10.75L11.25 10.75L11.25 20C11.25 20.4142 11.5858 20.75 12 20.75Z" fill="currentColor"></path>
                        <path d="M6.00002 10.75C5.69667 10.75 5.4232 10.5673 5.30711 10.287C5.19103 10.0068 5.25519 9.68417 5.46969 9.46967L11.4697 3.46967C11.6103 3.32902 11.8011 3.25 12 3.25C12.1989 3.25 12.3897 3.32902 12.5304 3.46967L18.5304 9.46967C18.7449 9.68417 18.809 10.0068 18.6929 10.287C18.5768 10.5673 18.3034 10.75 18 10.75L6.00002 10.75Z" fill="currentColor"></path>
                    </svg>
                </button>
            </template>
        </div>

        <div class="main-container min-h-screen text-black dark:text-white-dark" :class="[$store.app.navbar]">
            <!-- start sidebar section -->
        <?php include 'sidebar.php'; ?>
            <!-- end sidebar section -->

            <div class="main-content flex min-h-screen flex-col">
                <!-- start header section -->
                <header class="z-40" :class="{'dark' : $store.app.semidark && $store.app.menu === 'horizontal'}">
                    <div class="shadow-sm">
                        <div class="relative flex w-full items-center bg-white px-5 py-2.5 dark:bg-[#0e1726]">
                            <div class="horizontal-logo flex items-center justify-between ltr:mr-2 rtl:ml-2 lg:hidden">
                                <a href="index.php" class="main-logo flex shrink-0 items-center">
                                    <img class="inline w-8 ltr:-ml-1 rtl:-mr-1" src="assets/images/logo.png" alt="image">
                                    <span class="hidden align-middle text-2xl font-semibold transition-all duration-300 ltr:ml-1.5 rtl:mr-1.5 dark:text-white-light md:inline">Pluto Wifi</span>
                                </a>

                                <a href="javascript:;" class="collapse-icon flex flex-none rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary ltr:ml-2 rtl:mr-2 dark:bg-dark/40 dark:text-[#d0d2d6] dark:hover:bg-dark/60 dark:hover:text-primary lg:hidden" @click="$store.app.toggleSidebar()">
                                    <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M20 7L4 7" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        <path opacity="0.5" d="M20 12L4 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        <path d="M20 17L4 17" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                </a>
                            </div>
                            <div class="hidden ltr:mr-2 rtl:ml-2 sm:block">
                                <ul class="flex items-center space-x-2 rtl:space-x-reverse dark:text-[#d0d2d6]">
                                    <li>
                                        <a href="apps-todolist.php" class="block rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60">
                                            <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path opacity="0.5" d="M22 10.5V12C22 16.714 22 19.0711 20.5355 20.5355C19.0711 22 16.714 22 12 22C7.28595 22 4.92893 22 3.46447 20.5355C2 19.0711 2 16.714 2 12C2 7.28595 2 4.92893 3.46447 3.46447C4.92893 2 7.28595 2 12 2H13.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                <path d="M17.3009 2.80624L16.652 3.45506L10.6872 9.41993C10.2832 9.82394 10.0812 10.0259 9.90743 10.2487C9.70249 10.5114 9.52679 10.7957 9.38344 11.0965C9.26191 11.3515 9.17157 11.6225 8.99089 12.1646L8.41242 13.9L8.03811 15.0229C7.9492 15.2897 8.01862 15.5837 8.21744 15.7826C8.41626 15.9814 8.71035 16.0508 8.97709 15.9619L10.1 15.5876L11.8354 15.0091C12.3775 14.8284 12.6485 14.7381 12.9035 14.6166C13.2043 14.4732 13.4886 14.2975 13.7513 14.0926C13.9741 13.9188 14.1761 13.7168 14.5801 13.3128L20.5449 7.34795L21.1938 6.69914C22.2687 5.62415 22.2687 3.88124 21.1938 2.80624C20.1188 1.73125 18.3759 1.73125 17.3009 2.80624Z" stroke="currentColor" stroke-width="1.5"></path>
                                                <path opacity="0.5" d="M16.6522 3.45508C16.6522 3.45508 16.7333 4.83381 17.9499 6.05034C19.1664 7.26687 20.5451 7.34797 20.5451 7.34797M10.1002 15.5876L8.4126 13.9" stroke="currentColor" stroke-width="1.5"></path>
                                            </svg>
                                        </a>
                                    </li>
                                </ul>
                            </div>
                            <div x-data="header" class="flex items-center space-x-1.5 ltr:ml-auto rtl:mr-auto rtl:space-x-reverse dark:text-[#d0d2d6] sm:flex-1 ltr:sm:ml-0 sm:rtl:mr-0 lg:space-x-2">
                                <div class="sm:ltr:mr-auto sm:rtl:ml-auto" x-data="{ search: false }" @click.outside="search = false">
                                    <form class="absolute inset-x-0 top-1/2 z-10 mx-4 hidden -translate-y-1/2 sm:relative sm:top-0 sm:mx-0 sm:block sm:translate-y-0" :class="{'!block' : search}" @submit.prevent="search = false">
                                        <div class="relative">
                                            <input type="text" class="peer form-input bg-gray-100 placeholder:tracking-widest ltr:pl-9 ltr:pr-9 rtl:pr-9 rtl:pl-9 sm:bg-transparent ltr:sm:pr-4 rtl:sm:pl-4" placeholder="Search...">
                                            <button type="button" class="absolute inset-0 h-9 w-9 appearance-none peer-focus:text-primary ltr:right-auto rtl:left-auto">
                                                <svg class="mx-auto" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="11.5" cy="11.5" r="9.5" stroke="currentColor" stroke-width="1.5" opacity="0.5"></circle>
                                                    <path d="M18.5 18.5L22 22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                </svg>
                                            </button>
                                            <button type="button" class="absolute top-1/2 block -translate-y-1/2 hover:opacity-80 ltr:right-2 rtl:left-2 sm:hidden" @click="search = false">
                                                <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle opacity="0.5" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path d="M14.5 9.50002L9.5 14.5M9.49998 9.5L14.5 14.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </form>
                                    <button type="button" class="search_btn rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 dark:bg-dark/40 dark:hover:bg-dark/60 sm:hidden" @click="search = ! search">
                                        <svg class="mx-auto h-4.5 w-4.5 dark:text-[#d0d2d6]" width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="11.5" cy="11.5" r="9.5" stroke="currentColor" stroke-width="1.5" opacity="0.5"></circle>
                                            <path d="M18.5 18.5L22 22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </button>
                                </div>
                                <div>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'light'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('dark')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="12" cy="12" r="5" stroke="currentColor" stroke-width="1.5"></circle>
                                            <path d="M12 2V4" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M12 20V22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M4 12L2 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M22 12L20 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M19.7778 4.22266L17.5558 6.25424" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M4.22217 4.22266L6.44418 6.25424" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M6.44434 17.5557L4.22211 19.7779" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M19.7778 19.7773L17.5558 17.5551" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </a>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'dark'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('system')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M21.0672 11.8568L20.4253 11.469L21.0672 11.8568ZM12.1432 2.93276L11.7553 2.29085V2.29085L12.1432 2.93276ZM21.25 12C21.25 17.1086 17.1086 21.25 12 21.25V22.75C17.9371 22.75 22.75 17.9371 22.75 12H21.25ZM12 21.25C6.89137 21.25 2.75 17.1086 2.75 12H1.25C1.25 17.9371 6.06294 22.75 12 22.75V21.25ZM2.75 12C2.75 6.89137 6.89137 2.75 12 2.75V1.25C6.06294 1.25 1.25 6.06294 1.25 12H2.75ZM15.5 14.25C12.3244 14.25 9.75 11.6756 9.75 8.5H8.25C8.25 12.5041 11.4959 15.75 15.5 15.75V14.25ZM20.4253 11.469C19.4172 13.1373 17.5882 14.25 15.5 14.25V15.75C18.1349 15.75 20.4407 14.3439 21.7092 12.2447L20.4253 11.469ZM9.75 8.5C9.75 6.41182 10.8627 4.5828 12.531 3.57467L11.7553 2.29085C9.65609 3.5593 8.25 5.86509 8.25 8.5H9.75ZM12 2.75C11.9115 2.75 11.8077 2.71008 11.7324 2.63168C11.6686 2.56527 11.6538 2.50244 11.6503 2.47703C11.6461 2.44587 11.6482 2.35557 11.7553 2.29085L12.531 3.57467C13.0342 3.27065 13.196 2.71398 13.1368 2.27627C13.0754 1.82126 12.7166 1.25 12 1.25V2.75ZM21.7092 12.2447C21.6444 12.3518 21.5541 12.3539 21.523 12.3497C21.4976 12.3462 21.4347 12.3314 21.3683 12.2676C21.2899 12.1923 21.25 12.0885 21.25 12H22.75C22.75 11.2834 22.1787 10.9246 21.7237 10.8632C21.286 10.804 20.7293 10.9658 20.4253 11.469L21.7092 12.2447Z" fill="currentColor"></path>
                                        </svg>
                                    </a>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'system'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('light')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M3 9C3 6.17157 3 4.75736 3.87868 3.87868C4.75736 3 6.17157 3 9 3H15C17.8284 3 19.2426 3 20.1213 3.87868C21 4.75736 21 6.17157 21 9V14C21 15.8856 21 16.8284 20.4142 17.4142C19.8284 18 18.8856 18 17 18H7C5.11438 18 4.17157 18 3.58579 17.4142C3 16.8284 3 15.8856 3 14V9Z" stroke="currentColor" stroke-width="1.5"></path>
                                            <path opacity="0.5" d="M22 21H2" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M15 15H9" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </a>
                                </div>

                                <div class="dropdown shrink-0" x-data="dropdown" @click.outside="open = false">
                                    <a href="javascript:;" class="block rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="toggle">
                                        <img :src="`assets/images/flags/${$store.app.locale.toUpperCase()}.svg`" alt="image" class="h-5 w-5 rounded-full object-cover">
                                    </a>
                                    <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="top-11 grid w-[280px] grid-cols-2 gap-y-2 !px-2 font-semibold text-dark ltr:-right-14 rtl:-left-14 dark:text-white-dark dark:text-white-light/90 sm:ltr:-right-2 sm:rtl:-left-2">
                                        <template x-for="item in languages">
                                            <li>
                                                <a href="javascript:;" class="hover:text-primary" @click="$store.app.toggleLocale(item.value),toggle()" :class="{'bg-primary/10 text-primary' : $store.app.locale == item.value}">
                                                    <img class="h-5 w-5 rounded-full object-cover" :src="`assets/images/flags/${item.value.toUpperCase()}.svg`" alt="image">
                                                    <span class="ltr:ml-3 rtl:mr-3" x-text="item.key"></span>
                                                </a>
                                            </li>
                                        </template>
                                    </ul>
                                </div>

                                <div class="dropdown flex-shrink-0" x-data="dropdown" @click.outside="open = false">
                                    <a href="javascript:;" class="group relative" @click="toggle()">
                                        <span><img class="h-9 w-9 rounded-full object-cover saturate-50 group-hover:saturate-100" src="assets/images/user-profile.jpeg" alt="image"></span>
                                    </a>
                                    <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="top-11 w-[230px] !py-0 font-semibold text-dark ltr:right-0 rtl:left-0 dark:text-white-dark dark:text-white-light/90">
                                        <li>
                                            <div class="flex items-center px-4 py-4">
                                                <div class="flex-none">
                                                    <img class="h-10 w-10 rounded-md object-cover" src="assets/images/user-profile.jpeg" alt="image">
                                                </div>
                                                <div class="truncate ltr:pl-4 rtl:pr-4">
                                                    <h4 class="text-base">
                                                        Pluto Wifi
                                                </div>
                                            </div>
                                        </li>
                                        <li>
                                            <a href="users-profile.php" class="dark:hover:text-white" @click="toggle">
                                                <svg class="h-4.5 w-4.5 shrink-0 ltr:mr-2 rtl:ml-2" width="18" height="18" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="12" cy="6" r="4" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path opacity="0.5" d="M20 17.5C20 19.9853 20 22 12 22C4 22 4 19.9853 4 17.5C4 15.0147 7.58172 13 12 13C16.4183 13 20 15.0147 20 17.5Z" stroke="currentColor" stroke-width="1.5"></path>
                                                </svg>
                                                Profile</a>
                                        </li>
                                        <li class="border-t border-white-light dark:border-white-light/10">
                                            <a href="auth-boxed-signin.php" class="!py-3 text-danger" @click="toggle">
                                                <svg class="h-4.5 w-4.5 rotate-90 ltr:mr-2 rtl:ml-2" width="18" height="18" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path opacity="0.5" d="M17 9.00195C19.175 9.01406 20.3529 9.11051 21.1213 9.8789C22 10.7576 22 12.1718 22 15.0002V16.0002C22 18.8286 22 20.2429 21.1213 21.1215C20.2426 22.0002 18.8284 22.0002 16 22.0002H8C5.17157 22.0002 3.75736 22.0002 2.87868 21.1215C2 20.2429 2 18.8286 2 16.0002L2 15.0002C2 12.1718 2 10.7576 2.87868 9.87889C3.64706 9.11051 4.82497 9.01406 7 9.00195" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <path d="M12 15L12 2M12 2L15 5.5M12 2L9 5.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                                Sign Out
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <!-- horizontal menu -->
                        <ul class="horizontal-menu hidden border-t border-[#ebedf2] bg-white px-6 py-1.5 font-semibold text-black rtl:space-x-reverse dark:border-[#191e3a] dark:bg-[#0e1726] dark:text-white-dark lg:space-x-1.5 xl:space-x-8">
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link active">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M2 12.2039C2 9.91549 2 8.77128 2.5192 7.82274C3.0384 6.87421 3.98695 6.28551 5.88403 5.10813L7.88403 3.86687C9.88939 2.62229 10.8921 2 12 2C13.1079 2 14.1106 2.62229 16.116 3.86687L18.116 5.10812C20.0131 6.28551 20.9616 6.87421 21.4808 7.82274C22 8.77128 22 9.91549 22 12.2039V13.725C22 17.6258 22 19.5763 20.8284 20.7881C19.6569 22 17.7712 22 14 22H10C6.22876 22 4.34315 22 3.17157 20.7881C2 19.5763 2 17.6258 2 13.725V12.2039Z" fill="currentColor"></path>
                                            <path d="M9 17.25C8.58579 17.25 8.25 17.5858 8.25 18C8.25 18.4142 8.58579 18.75 9 18.75H15C15.4142 18.75 15.75 18.4142 15.75 18C15.75 17.5858 15.4142 17.25 15 17.25H9Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Dashboard</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="index.php">Sales</a>
                                    </li>
                                    <li>
                                        <a href="finance.php" class="active">Finance</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <g opacity="0.5">
                                                <path d="M14 2.75C15.9068 2.75 17.2615 2.75159 18.2892 2.88976C19.2952 3.02503 19.8749 3.27869 20.2981 3.7019C20.7213 4.12511 20.975 4.70476 21.1102 5.71085C21.2484 6.73851 21.25 8.09318 21.25 10C21.25 10.4142 21.5858 10.75 22 10.75C22.4142 10.75 22.75 10.4142 22.75 10V9.94359C22.75 8.10583 22.75 6.65019 22.5969 5.51098C22.4392 4.33856 22.1071 3.38961 21.3588 2.64124C20.6104 1.89288 19.6614 1.56076 18.489 1.40314C17.3498 1.24997 15.8942 1.24998 14.0564 1.25H14C13.5858 1.25 13.25 1.58579 13.25 2C13.25 2.41421 13.5858 2.75 14 2.75Z" fill="currentColor"></path>
                                                <path d="M9.94358 1.25H10C10.4142 1.25 10.75 1.58579 10.75 2C10.75 2.41421 10.4142 2.75 10 2.75C8.09318 2.75 6.73851 2.75159 5.71085 2.88976C4.70476 3.02503 4.12511 3.27869 3.7019 3.7019C3.27869 4.12511 3.02503 4.70476 2.88976 5.71085C2.75159 6.73851 2.75 8.09318 2.75 10C2.75 10.4142 2.41421 10.75 2 10.75C1.58579 10.75 1.25 10.4142 1.25 10V9.94358C1.24998 8.10583 1.24997 6.65019 1.40314 5.51098C1.56076 4.33856 1.89288 3.38961 2.64124 2.64124C3.38961 1.89288 4.33856 1.56076 5.51098 1.40314C6.65019 1.24997 8.10583 1.24998 9.94358 1.25Z" fill="currentColor"></path>
                                                <path d="M22 13.25C22.4142 13.25 22.75 13.5858 22.75 14V14.0564C22.75 15.8942 22.75 17.3498 22.5969 18.489C22.4392 19.6614 22.1071 20.6104 21.3588 21.3588C20.6104 22.1071 19.6614 22.4392 18.489 22.5969C17.3498 22.75 15.8942 22.75 14.0564 22.75H14C13.5858 22.75 13.25 22.4142 13.25 22C13.25 21.5858 13.5858 21.25 14 21.25C15.9068 21.25 17.2615 21.2484 18.2892 21.1102C19.2952 20.975 19.8749 20.7213 20.2981 20.2981C20.7213 19.8749 20.975 19.2952 21.1102 18.2892C21.2484 17.2615 21.25 15.9068 21.25 14C21.25 13.5858 21.5858 13.25 22 13.25Z" fill="currentColor"></path>
                                                <path d="M2.75 14C2.75 13.5858 2.41421 13.25 2 13.25C1.58579 13.25 1.25 13.5858 1.25 14V14.0564C1.24998 15.8942 1.24997 17.3498 1.40314 18.489C1.56076 19.6614 1.89288 20.6104 2.64124 21.3588C3.38961 22.1071 4.33856 22.4392 5.51098 22.5969C6.65019 22.75 8.10583 22.75 9.94359 22.75H10C10.4142 22.75 10.75 22.4142 10.75 22C10.75 21.5858 10.4142 21.25 10 21.25C8.09318 21.25 6.73851 21.2484 5.71085 21.1102C4.70476 20.975 4.12511 20.7213 3.7019 20.2981C3.27869 19.8749 3.02503 19.2952 2.88976 18.2892C2.75159 17.2615 2.75 15.9068 2.75 14Z" fill="currentColor"></path>
                                            </g>
                                            <path d="M5.52721 5.52721C5 6.05442 5 6.90294 5 8.6C5 9.73137 5 10.2971 5.35147 10.6485C5.70294 11 6.26863 11 7.4 11H8.6C9.73137 11 10.2971 11 10.6485 10.6485C11 10.2971 11 9.73137 11 8.6V7.4C11 6.26863 11 5.70294 10.6485 5.35147C10.2971 5 9.73137 5 8.6 5C6.90294 5 6.05442 5 5.52721 5.52721Z" fill="currentColor"></path>
                                            <path d="M5.52721 18.4728C5 17.9456 5 17.0971 5 15.4C5 14.2686 5 13.7029 5.35147 13.3515C5.70294 13 6.26863 13 7.4 13H8.6C9.73137 13 10.2971 13 10.6485 13.3515C11 13.7029 11 14.2686 11 15.4V16.6C11 17.7314 11 18.2971 10.6485 18.6485C10.2971 19 9.73138 19 8.60002 19C6.90298 19 6.05441 19 5.52721 18.4728Z" fill="currentColor"></path>
                                            <path d="M13 7.4C13 6.26863 13 5.70294 13.3515 5.35147C13.7029 5 14.2686 5 15.4 5C17.0971 5 17.9456 5 18.4728 5.52721C19 6.05442 19 6.90294 19 8.6C19 9.73137 19 10.2971 18.6485 10.6485C18.2971 11 17.7314 11 16.6 11H15.4C14.2686 11 13.7029 11 13.3515 10.6485C13 10.2971 13 9.73137 13 8.6V7.4Z" fill="currentColor"></path>
                                            <path d="M13.3515 18.6485C13 18.2971 13 17.7314 13 16.6V15.4C13 14.2686 13 13.7029 13.3515 13.3515C13.7029 13 14.2686 13 15.4 13H16.6C17.7314 13 18.2971 13 18.6485 13.3515C19 13.7029 19 14.2686 19 15.4C19 17.097 19 17.9456 18.4728 18.4728C17.9456 19 17.0971 19 15.4 19C14.2687 19 13.7029 19 13.3515 18.6485Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Apps</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="apps-chat.php">Chat</a>
                                    </li>
                                    <li>
                                        <a href="apps-mailbox.php">Mailbox</a>
                                    </li>
                                    <li>
                                        <a href="apps-todolist.php">Todo List</a>
                                    </li>
                                    <li>
                                        <a href="apps-notes.php">Notes</a>
                                    </li>
                                    <li>
                                        <a href="apps-scrumboard.php">Scrumboard</a>
                                    </li>
                                    <li>
                                        <a href="apps-contacts.php">Contacts</a>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Invoice
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="apps-invoice-list.php">List</a>
                                            </li>
                                            <li>
                                                <a href="apps-invoice-preview.php">Preview</a>
                                            </li>
                                            <li>
                                                <a href="apps-invoice-add.php">Add</a>
                                            </li>
                                            <li>
                                                <a href="apps-invoice-edit.php">Edit</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li>
                                        <a href="apps-calendar.php">Calendar</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path d="M8.42229 20.6181C10.1779 21.5395 11.0557 22.0001 12 22.0001V12.0001L2.63802 7.07275C2.62423 7.09491 2.6107 7.11727 2.5974 7.13986C2 8.15436 2 9.41678 2 11.9416V12.0586C2 14.5834 2 15.8459 2.5974 16.8604C3.19479 17.8749 4.27063 18.4395 6.42229 19.5686L8.42229 20.6181Z" fill="currentColor"></path>
                                            <path opacity="0.7" d="M17.5774 4.43152L15.5774 3.38197C13.8218 2.46066 12.944 2 11.9997 2C11.0554 2 10.1776 2.46066 8.42197 3.38197L6.42197 4.43152C4.31821 5.53552 3.24291 6.09982 2.6377 7.07264L11.9997 12L21.3617 7.07264C20.7564 6.09982 19.6811 5.53552 17.5774 4.43152Z" fill="currentColor"></path>
                                            <path opacity="0.5" d="M21.4026 7.13986C21.3893 7.11727 21.3758 7.09491 21.362 7.07275L12 12.0001V22.0001C12.9443 22.0001 13.8221 21.5395 15.5777 20.6181L17.5777 19.5686C19.7294 18.4395 20.8052 17.8749 21.4026 16.8604C22 15.8459 22 14.5834 22 12.0586V11.9416C22 9.41678 22 8.15436 21.4026 7.13986Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Components</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="components-tabs.php">Tabs</a>
                                    </li>
                                    <li>
                                        <a href="components-accordions.php">Accordions</a>
                                    </li>
                                    <li>
                                        <a href="components-modals.php">Modals</a>
                                    </li>
                                    <li>
                                        <a href="components-cards.php">Cards</a>
                                    </li>
                                    <li>
                                        <a href="components-carousel.php">Carousel</a>
                                    </li>
                                    <li>
                                        <a href="components-countdown.php">Countdown</a>
                                    </li>
                                    <li>
                                        <a href="components-counter.php">Counter</a>
                                    </li>
                                    <li>
                                        <a href="components-sweetalert.php">Sweet Alerts</a>
                                    </li>
                                    <li>
                                        <a href="components-timeline.php">Timeline</a>
                                    </li>
                                    <li>
                                        <a href="components-notifications.php">Notifications</a>
                                    </li>
                                    <li>
                                        <a href="components-media-object.php">Media Object</a>
                                    </li>
                                    <li>
                                        <a href="components-list-group.php">List Group</a>
                                    </li>
                                    <li>
                                        <a href="components-pricing-table.php">Pricing Tables</a>
                                    </li>
                                    <li>
                                        <a href="components-lightbox.php">Lightbox</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M8.73167 5.77133L5.66953 9.91436C4.3848 11.6526 3.74244 12.5217 4.09639 13.205C4.10225 13.2164 4.10829 13.2276 4.1145 13.2387C4.48945 13.9117 5.59888 13.9117 7.81775 13.9117C9.05079 13.9117 9.6673 13.9117 10.054 14.2754L10.074 14.2946L13.946 9.72466L13.926 9.70541C13.5474 9.33386 13.5474 8.74151 13.5474 7.55682V7.24712C13.5474 3.96249 13.5474 2.32018 12.6241 2.03721C11.7007 1.75425 10.711 3.09327 8.73167 5.77133Z" fill="currentColor"></path>
                                            <path opacity="0.5" d="M10.4527 16.4432L10.4527 16.7528C10.4527 20.0374 10.4527 21.6798 11.376 21.9627C12.2994 22.2457 13.2891 20.9067 15.2685 18.2286L18.3306 14.0856C19.6154 12.3474 20.2577 11.4783 19.9038 10.7949C19.8979 10.7836 19.8919 10.7724 19.8857 10.7613C19.5107 10.0883 18.4013 10.0883 16.1824 10.0883C14.9494 10.0883 14.3329 10.0883 13.9462 9.72461L10.0742 14.2946C10.4528 14.6661 10.4527 15.2585 10.4527 16.4432Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Elements</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="elements-alerts.php">Alerts</a>
                                    </li>
                                    <li>
                                        <a href="elements-avatar.php">Avatar</a>
                                    </li>
                                    <li>
                                        <a href="elements-badges.php">Badges</a>
                                    </li>
                                    <li>
                                        <a href="elements-breadcrumbs.php">Breadcrumbs</a>
                                    </li>
                                    <li>
                                        <a href="elements-buttons.php">Buttons</a>
                                    </li>
                                    <li>
                                        <a href="elements-buttons-group.php">Button Groups</a>
                                    </li>
                                    <li>
                                        <a href="elements-color-library.php">Color Library</a>
                                    </li>
                                    <li>
                                        <a href="elements-dropdown.php">Dropdown</a>
                                    </li>
                                    <li>
                                        <a href="elements-infobox.php">Infobox</a>
                                    </li>
                                    <li>
                                        <a href="elements-jumbotron.php">Jumbotron</a>
                                    </li>
                                    <li>
                                        <a href="elements-loader.php">Loader</a>
                                    </li>
                                    <li>
                                        <a href="elements-pagination.php">Pagination</a>
                                    </li>
                                    <li>
                                        <a href="elements-popovers.php">Popovers</a>
                                    </li>
                                    <li>
                                        <a href="elements-progress-bar.php">Progress Bar</a>
                                    </li>
                                    <li>
                                        <a href="elements-search.php">Search</a>
                                    </li>
                                    <li>
                                        <a href="elements-tooltips.php">Tooltips</a>
                                    </li>
                                    <li>
                                        <a href="elements-treeview.php">Treeview</a>
                                    </li>
                                    <li>
                                        <a href="elements-typography.php">Typography</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path d="M4.97883 9.68508C2.99294 8.89073 2 8.49355 2 8C2 7.50645 2.99294 7.10927 4.97883 6.31492L7.7873 5.19153C9.77318 4.39718 10.7661 4 12 4C13.2339 4 14.2268 4.39718 16.2127 5.19153L19.0212 6.31492C21.0071 7.10927 22 7.50645 22 8C22 8.49355 21.0071 8.89073 19.0212 9.68508L16.2127 10.8085C14.2268 11.6028 13.2339 12 12 12C10.7661 12 9.77318 11.6028 7.7873 10.8085L4.97883 9.68508Z" fill="currentColor"></path>
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M2 8C2 8.49355 2.99294 8.89073 4.97883 9.68508L7.7873 10.8085C9.77318 11.6028 10.7661 12 12 12C13.2339 12 14.2268 11.6028 16.2127 10.8085L19.0212 9.68508C21.0071 8.89073 22 8.49355 22 8C22 7.50645 21.0071 7.10927 19.0212 6.31492L16.2127 5.19153C14.2268 4.39718 13.2339 4 12 4C10.7661 4 9.77318 4.39718 7.7873 5.19153L4.97883 6.31492C2.99294 7.10927 2 7.50645 2 8Z" fill="currentColor"></path>
                                            <path opacity="0.7" d="M5.76613 10L4.97883 10.3149C2.99294 11.1093 2 11.5065 2 12C2 12.4935 2.99294 12.8907 4.97883 13.6851L7.7873 14.8085C9.77318 15.6028 10.7661 16 12 16C13.2339 16 14.2268 15.6028 16.2127 14.8085L19.0212 13.6851C21.0071 12.8907 22 12.4935 22 12C22 11.5065 21.0071 11.1093 19.0212 10.3149L18.2339 10L16.2127 10.8085C14.2268 11.6028 13.2339 12 12 12C10.7661 12 9.77318 11.6028 7.7873 10.8085L5.76613 10Z" fill="currentColor"></path>
                                            <path opacity="0.4" d="M5.76613 14L4.97883 14.3149C2.99294 15.1093 2 15.5065 2 16C2 16.4935 2.99294 16.8907 4.97883 17.6851L7.7873 18.8085C9.77318 19.6028 10.7661 20 12 20C13.2339 20 14.2268 19.6028 16.2127 18.8085L19.0212 17.6851C21.0071 16.8907 22 16.4935 22 16C22 15.5065 21.0071 15.1093 19.0212 14.3149L18.2339 14L16.2127 14.8085C14.2268 15.6028 13.2339 16 12 16C10.7661 16 9.77318 15.6028 7.7873 14.8085L5.76613 14Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Tables</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="tables.php">Tables</a>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Data Tables
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="datatables-basic.php">Basic</a>
                                            </li>
                                            <li>
                                                <a href="datatables-advanced.php">Advanced</a>
                                            </li>
                                            <li>
                                                <a href="datatables-skin.php">Skin</a>
                                            </li>
                                            <li>
                                                <a href="datatables-order-sorting.php">Order Sorting</a>
                                            </li>
                                            <li>
                                                <a href="datatables-multi-column.php">Multi Column</a>
                                            </li>
                                            <li>
                                                <a href="datatables-multiple-tables.php">Multiple Tables</a>
                                            </li>
                                            <li>
                                                <a href="datatables-alt-pagination.php">Alt. Pagination</a>
                                            </li>
                                            <li>
                                                <a href="datatables-checkbox.php">Checkbox</a>
                                            </li>
                                            <li>
                                                <a href="datatables-range-search.php">Range Search</a>
                                            </li>
                                            <li>
                                                <a href="datatables-export.php">Export</a>
                                            </li>
                                            <li>
                                                <a href="datatables-sticky-header.php">Sticky Header</a>
                                            </li>
                                            <li>
                                                <a href="datatables-clone-header.php">Clone Header</a>
                                            </li>
                                            <li>
                                                <a href="datatables-column-chooser.php">Column Chooser</a>
                                            </li>
                                        </ul>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M3 10C3 6.22876 3 4.34315 4.17157 3.17157C5.34315 2 7.22876 2 11 2H13C16.7712 2 18.6569 2 19.8284 3.17157C21 4.34315 21 6.22876 21 10V14C21 17.7712 21 19.6569 19.8284 20.8284C18.6569 22 16.7712 22 13 22H11C7.22876 22 5.34315 22 4.17157 20.8284C3 19.6569 3 17.7712 3 14V10Z" fill="currentColor"></path>
                                            <path d="M16.5189 16.5013C16.6939 16.3648 16.8526 16.2061 17.1701 15.8886L21.1275 11.9312C21.2231 11.8356 21.1793 11.6708 21.0515 11.6264C20.5844 11.4644 19.9767 11.1601 19.4083 10.5917C18.8399 10.0233 18.5356 9.41561 18.3736 8.94849C18.3292 8.82066 18.1644 8.77687 18.0688 8.87254L14.1114 12.8299C13.7939 13.1474 13.6352 13.3061 13.4987 13.4811C13.3377 13.6876 13.1996 13.9109 13.087 14.1473C12.9915 14.3476 12.9205 14.5606 12.7786 14.9865L12.5951 15.5368L12.3034 16.4118L12.0299 17.2323C11.9601 17.4419 12.0146 17.6729 12.1708 17.8292C12.3271 17.9854 12.5581 18.0399 12.7677 17.9701L13.5882 17.6966L14.4632 17.4049L15.0135 17.2214L15.0136 17.2214C15.4394 17.0795 15.6524 17.0085 15.8527 16.913C16.0891 16.8004 16.3124 16.6623 16.5189 16.5013Z" fill="currentColor"></path>
                                            <path d="M22.3665 10.6922C23.2112 9.84754 23.2112 8.47812 22.3665 7.63348C21.5219 6.78884 20.1525 6.78884 19.3078 7.63348L19.1806 7.76071C19.0578 7.88348 19.0022 8.05496 19.0329 8.22586C19.0522 8.33336 19.0879 8.49053 19.153 8.67807C19.2831 9.05314 19.5288 9.54549 19.9917 10.0083C20.4545 10.4712 20.9469 10.7169 21.3219 10.847C21.5095 10.9121 21.6666 10.9478 21.7741 10.9671C21.945 10.9978 22.1165 10.9422 22.2393 10.8194L22.3665 10.6922Z" fill="currentColor"></path>
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M7.25 9C7.25 8.58579 7.58579 8.25 8 8.25H14.5C14.9142 8.25 15.25 8.58579 15.25 9C15.25 9.41421 14.9142 9.75 14.5 9.75H8C7.58579 9.75 7.25 9.41421 7.25 9ZM7.25 13C7.25 12.5858 7.58579 12.25 8 12.25H11C11.4142 12.25 11.75 12.5858 11.75 13C11.75 13.4142 11.4142 13.75 11 13.75H8C7.58579 13.75 7.25 13.4142 7.25 13ZM7.25 17C7.25 16.5858 7.58579 16.25 8 16.25H9.5C9.91421 16.25 10.25 16.5858 10.25 17C10.25 17.4142 9.91421 17.75 9.5 17.75H8C7.58579 17.75 7.25 17.4142 7.25 17Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Settings</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="forms-basic.php">Basic</a>
                                    </li>
                                    <li>
                                        <a href="forms-input-group.php">Input Group</a>
                                    </li>
                                    <li>
                                        <a href="forms-layouts.php">Layouts</a>
                                    </li>
                                    <li>
                                        <a href="forms-validation.php">Validation</a>
                                    </li>
                                    <li>
                                        <a href="forms-input-mask.php">Input Mask</a>
                                    </li>
                                    <li>
                                        <a href="forms-select2.php">Select2</a>
                                    </li>
                                    <li>
                                        <a href="forms-touchspin.php">TouchSpin</a>
                                    </li>
                                    <li>
                                        <a href="forms-checkbox-radio.php">Checkbox & Radio</a>
                                    </li>
                                    <li>
                                        <a href="forms-switches.php">Switches</a>
                                    </li>
                                    <li>
                                        <a href="forms-wizards.php">Wizards</a>
                                    </li>
                                    <li>
                                        <a href="forms-file-upload.php">File Upload</a>
                                    </li>
                                    <li>
                                        <a href="forms-quill-editor.php">Quill Editor</a>
                                    </li>
                                    <li>
                                        <a href="forms-markdown-editor.php">Markdown Editor</a>
                                    </li>
                                    <li>
                                        <a href="forms-date-picker.php">Date & Range Picker</a>
                                    </li>
                                    <li>
                                        <a href="forms-clipboard.php">Clipboard</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" fill-rule="evenodd" clip-rule="evenodd" d="M14 22H10C6.22876 22 4.34315 22 3.17157 20.8284C2 19.6569 2 17.7712 2 14V10C2 6.22876 2 4.34315 3.17157 3.17157C4.34315 2 6.23869 2 10.0298 2C10.6358 2 11.1214 2 11.53 2.01666C11.5166 2.09659 11.5095 2.17813 11.5092 2.26057L11.5 5.09497C11.4999 6.19207 11.4998 7.16164 11.6049 7.94316C11.7188 8.79028 11.9803 9.63726 12.6716 10.3285C13.3628 11.0198 14.2098 11.2813 15.0569 11.3952C15.8385 11.5003 16.808 11.5002 17.9051 11.5001L18 11.5001H21.9574C22 12.0344 22 12.6901 22 13.5629V14C22 17.7712 22 19.6569 20.8284 20.8284C19.6569 22 17.7712 22 14 22Z" fill="currentColor"></path>
                                            <path d="M6 13.75C5.58579 13.75 5.25 14.0858 5.25 14.5C5.25 14.9142 5.58579 15.25 6 15.25H14C14.4142 15.25 14.75 14.9142 14.75 14.5C14.75 14.0858 14.4142 13.75 14 13.75H6Z" fill="currentColor"></path>
                                            <path d="M6 17.25C5.58579 17.25 5.25 17.5858 5.25 18C5.25 18.4142 5.58579 18.75 6 18.75H11.5C11.9142 18.75 12.25 18.4142 12.25 18C12.25 17.5858 11.9142 17.25 11.5 17.25H6Z" fill="currentColor"></path>
                                            <path d="M11.5092 2.2601L11.5 5.0945C11.4999 6.1916 11.4998 7.16117 11.6049 7.94269C11.7188 8.78981 11.9803 9.6368 12.6716 10.3281C13.3629 11.0193 14.2098 11.2808 15.057 11.3947C15.8385 11.4998 16.808 11.4997 17.9051 11.4996L21.9574 11.4996C21.9698 11.6552 21.9786 11.821 21.9848 11.9995H22C22 11.732 22 11.5983 21.9901 11.4408C21.9335 10.5463 21.5617 9.52125 21.0315 8.79853C20.9382 8.6713 20.8743 8.59493 20.7467 8.44218C19.9542 7.49359 18.911 6.31193 18 5.49953C17.1892 4.77645 16.0787 3.98536 15.1101 3.3385C14.2781 2.78275 13.862 2.50487 13.2915 2.29834C13.1403 2.24359 12.9408 2.18311 12.7846 2.14466C12.4006 2.05013 12.0268 2.01725 11.5 2.00586L11.5092 2.2601Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Pages</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li class="relative">
                                        <a href="javascript:;">Users
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="users-profile.php">Profile</a>
                                            </li>
                                            <li>
                                                <a href="users-account-settings.php">Account Settings</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li>
                                        <a href="pages-knowledge-base.php">Knowledge Base</a>
                                    </li>
                                    <li>
                                        <a href="pages-contact-us-boxed.php" target="_blank">Contact Us Boxed</a>
                                    </li>
                                    <li>
                                        <a href="pages-contact-us-cover.php" target="_blank">Contact Us Cover</a>
                                    </li>
                                    <li>
                                        <a href="pages-faq.php">Faq</a>
                                    </li>
                                    <li>
                                        <a href="pages-coming-soon-boxed.php" target="_blank">Coming Soon Boxed</a>
                                    </li>
                                    <li>
                                        <a href="pages-coming-soon-cover.php" target="_blank">Coming Soon Cover</a>
                                    </li>
                                    <li>
                                        <a href="pages-maintenence.php" target="_blank">Maintanence</a>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Error
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="pages-error404.php" target="_blank">404</a>
                                            </li>
                                            <li>
                                                <a href="pages-error500.php" target="_blank">500</a>
                                            </li>
                                            <li>
                                                <a href="pages-error503.php" target="_blank">503</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Login
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="auth-cover-login.php" target="_blank">Login Cover</a>
                                            </li>
                                            <li>
                                                <a href="auth-boxed-signin.php" target="_blank">Login Boxed</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Register
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="auth-cover-register.php" target="_blank">Register Cover</a>
                                            </li>
                                            <li>
                                                <a href="auth-boxed-signup.php" target="_blank">Register Boxed</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Password Recovery
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="auth-cover-password-reset.php" target="_blank">Recover ID Cover</a>
                                            </li>
                                            <li>
                                                <a href="auth-boxed-password-reset.php" target="_blank">Recover ID Boxed</a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li class="relative">
                                        <a href="javascript:;">Lockscreen
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="auth-cover-lockscreen.php" target="_blank">Unlock Cover</a>
                                            </li>
                                            <li>
                                                <a href="auth-boxed-lockscreen.php" target="_blank">Unlock Boxed</a>
                                            </li>
                                        </ul>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M22 12C22 17.5228 17.5228 22 12 22C6.47715 22 2 17.5228 2 12C2 6.47715 6.47715 2 12 2C17.5228 2 22 6.47715 22 12Z" fill="currentColor"></path>
                                            <path d="M12.75 9C12.75 8.58579 12.4142 8.25 12 8.25C11.5858 8.25 11.25 8.58579 11.25 9L11.25 11.25H9C8.58579 11.25 8.25 11.5858 8.25 12C8.25 12.4142 8.58579 12.75 9 12.75H11.25V15C11.25 15.4142 11.5858 15.75 12 15.75C12.4142 15.75 12.75 15.4142 12.75 15L12.75 12.75H15C15.4142 12.75 15.75 12.4142 15.75 12C15.75 11.5858 15.4142 11.25 15 11.25H12.75V9Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">More</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="dragndrop.php">Drag and Drop</a>
                                    </li>
                                    <!-- <li>
                                        <a href="charts.php">Charts</a>
                                    </li> -->
                                    <li>
                                        <a href="font-icons.php">Font Icons</a>
                                    </li>
                                    <li>
                                        <a href="widgets.php">Widgets</a>
                                    </li>
                                    <li>
                                        <a href="souysoeng.com" target="_blank">Documentation</a>
                                    </li>
                                </ul>
                            </li>
                        </ul>
                    </div>
                </header>
                <!-- end header section -->

                <div class="animate__animated p-6" :class="[$store.app.animation]">
                    <!-- start main content section -->
                    <div x-data="sales">
                        <ul class="flex space-x-2 rtl:space-x-reverse">
                            <li>
                                <a href="javascript:;" class="text-primary hover:underline">Dashboard</a>
                            </li>
                            <li class="before:content-['/'] ltr:before:mr-1 rtl:before:ml-1">
                                <span>Sales</span>
                            </li>
                        </ul>

                        <div class="pt-5">
                            <div class="mb-6 grid gap-6 xl:grid-cols-3">
                                

                                <div class="panel h-full xl:col-span-2">
                                    <!-- overlay -->
                                    <form method="GET" class="search-form-overlay relative border border-white-dark/20 rounded-md h-12 w-full mb-4">
                                        <input type="text" name="search_user" value="<?= htmlspecialchars($_GET['search_user'] ?? '') ?>" placeholder="Search by username or momo number..." class="form-input bg-white h-full placeholder:tracking-wider ltr:pl-12 rtl:pr-12 peer" />
                                        <button type="submit" class="text-dark/70 absolute ltr:right-1 rtl:left-1 inset-y-0 my-auto w-9 h-9 p-0 flex items-center justify-center peer-focus:text-primary">
                                            <svg width="20" height="20" viewBox="0 0 24 24"><circle cx="11" cy="11" r="8" stroke="currentColor" stroke-width="2" fill="none"/><line x1="21" y1="21" x2="16.65" y2="16.65" stroke="currentColor" stroke-width="2" stroke-linecap="round"/></svg>
                                        </button>
                                    </form>
                                    <div class="mb-5 flex items-center dark:text-white-light">
                                        <h5 class="text-lg font-semibold">Revenue</h5>
                                        <div x-data="dropdown" @click.outside="open = false" class="dropdown ltr:ml-auto rtl:mr-auto">
                                            <a href="javascript:;" @click="toggle">
                                                <svg class="h-5 w-5 text-black/70 hover:!text-primary dark:text-white/70" viewBox="0 0 24 24" fill="none">
                                                    <circle cx="5" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle opacity="0.5" cx="12" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle cx="19" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                </svg>
                                            </a>
                                            <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="ltr:right-0 rtl:left-0">
                                                <li><a href="javascript:;" @click="setMode('weekly');toggle()">Weekly</a></li>
                                                <li><a href="javascript:;" @click="setMode('monthly');toggle()">Monthly</a></li>
                                                <li><a href="javascript:;" @click="setMode('yearly');toggle()">Yearly</a></li>
                                            </ul>
                                        </div>
                                    </div>
                                    <div id="revenue-totals" class="mb-2 text-lg dark:text-white-light/90">
                                        <span x-text="$store.sales.mode === 'weekly' ? 'Total Revenue This Week: GH¢ <?= number_format($total_this_week,2) ?> | Last Week: GH¢ <?= number_format($total_last_week,2) ?>' : ($store.sales.mode === 'monthly' ? 'Total Revenue This Month: GH¢ <?= number_format($total_this_month,2) ?> | Last Month: GH¢ <?= number_format($total_last_month,2) ?>' : 'Total Revenue This Year: GH¢ <?= number_format($total_this_year,2) ?> | Last Year: GH¢ <?= number_format($total_last_year,2) ?>')"></span>
                                    </div>
                                    <div class="relative overflow-hidden">
                                        <div x-ref="revenueChart" class="rounded-lg bg-white dark:bg-black">
                                            <!-- loader -->
                                            <div class="grid min-h-[325px] place-content-center bg-white-light/30 dark:bg-dark dark:bg-opacity-[0.08]">
                                                <span class="inline-flex h-5 w-5 animate-spin rounded-full border-2 border-black !border-l-transparent dark:border-white"></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="panel h-full">
                                    <div class="mb-5 flex items-center">
                                        <h5 class="text-lg font-semibold dark:text-white-light">Daily Sales By Profile</h5>
                                    </div>
                                    <div class="overflow-hidden">
                                        <div x-ref="salesByCategory" class="rounded-lg bg-white dark:bg-black">
                                            <!-- loader -->
                                            <div class="grid min-h-[353px] place-content-center bg-white-light/30 dark:bg-dark dark:bg-opacity-[0.08]">
                                                <span class="inline-flex h-5 w-5 animate-spin rounded-full border-2 border-black !border-l-transparent dark:border-white"></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-6 grid gap-6 sm:grid-cols-2 xl:grid-cols-3">
                                <div class="panel h-full sm:col-span-2 xl:col-span-1">
                                    <div class="mb-5 flex items-center">
                                        <h5 class="text-lg font-semibold dark:text-white-light">
                                            Daily Sales <span class="block text-sm font-normal text-white-dark">Go to columns for details.</span>
                                        </h5>
                                        <div class="relative ltr:ml-auto rtl:mr-auto">
                                            <div class="grid h-11 w-11 place-content-center rounded-full bg-[#ffeccb] text-warning dark:bg-warning dark:text-[#ffeccb]">
                                                <svg width="40" height="40" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M12 6V18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <path d="M15 9.5C15 8.11929 13.6569 7 12 7C10.3431 7 9 8.11929 9 9.5C9 10.8807 10.3431 12 12 12C13.6569 12 15 13.1193 15 14.5C15 15.8807 13.6569 17 12 17C10.3431 17 9 15.8807 9 14.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="overflow-hidden">
                                        <div x-ref="dailySales" class="rounded-lg bg-white dark:bg-black">
                                            <!-- loader -->
                                            <div class="grid min-h-[175px] place-content-center bg-white-light/30 dark:bg-dark dark:bg-opacity-[0.08]">
                                                <span class="inline-flex h-5 w-5 animate-spin rounded-full border-2 border-black !border-l-transparent dark:border-white"></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="panel h-full">
                                    <div class="mb-5 flex items-center dark:text-white-light">
                                        <h5 class="text-lg font-semibold">Summary</h5>
                                        <div x-data="dropdown" @click.outside="open = false" class="dropdown ltr:ml-auto rtl:mr-auto">
                                            <a href="javascript:;" @click="toggle">
                                                <svg class="h-5 w-5 text-black/70 hover:!text-primary dark:text-white/70" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="5" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle opacity="0.5" cx="12" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle cx="19" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                </svg>
                                            </a>
                                            <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="ltr:right-0 rtl:left-0">
                                                <li><a href="javascript:;" @click="toggle">View Report</a></li>
                                                <li><a href="javascript:;" @click="toggle">Edit Report</a></li>
                                                <li><a href="javascript:;" @click="toggle">Mark as Done</a></li>
                                            </ul>
                                        </div>
                                    </div>
                                    <div class="space-y-9">
                                        <div class="flex items-center">
                                            <div class="h-9 w-9 ltr:mr-3 rtl:ml-3">
                                                <div class="grid h-9 w-9 place-content-center rounded-full bg-secondary-light text-secondary dark:bg-secondary dark:text-secondary-light">
                                                    <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <path d="M3.74157 18.5545C4.94119 20 7.17389 20 11.6393 20H12.3605C16.8259 20 19.0586 20 20.2582 18.5545M3.74157 18.5545C2.54194 17.1091 2.9534 14.9146 3.77633 10.5257C4.36155 7.40452 4.65416 5.84393 5.76506 4.92196M3.74157 18.5545C3.74156 18.5545 3.74157 18.5545 3.74157 18.5545ZM20.2582 18.5545C21.4578 17.1091 21.0464 14.9146 20.2235 10.5257C19.6382 7.40452 19.3456 5.84393 18.2347 4.92196M20.2582 18.5545C20.2582 18.5545 20.2582 18.5545 20.2582 18.5545ZM18.2347 4.92196C17.1238 4 15.5361 4 12.3605 4H11.6393C8.46374 4 6.87596 4 5.76506 4.92196M18.2347 4.92196C18.2347 4.92196 18.2347 4.92196 18.2347 4.92196ZM5.76506 4.92196C5.76506 4.92196 5.76506 4.92196 5.76506 4.92196Z" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path opacity="0.5" d="M9.1709 8C9.58273 9.16519 10.694 10 12.0002 10C13.3064 10 14.4177 9.16519 14.8295 8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                            <div class="flex-1">
                                                <div class="mb-2 flex font-semibold text-white-dark">
                                                    <h6>Total Attempted Transactions</h6>
                                                    <p class="ltr:ml-auto rtl:mr-auto">GH¢ <?= number_format($totalAttempted, 2) ?></p>
                                                </div>          
                                                <div class="h-2 rounded-full bg-dark-light shadow dark:bg-[#1b2e4b]">
                                                    <div class="h-full w-11/12 rounded-full bg-gradient-to-r from-[#7579ff] to-[#b224ef]"></div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="flex items-center">
                                            <div class="h-9 w-9 ltr:mr-3 rtl:ml-3">
                                                <div class="grid h-9 w-9 place-content-center rounded-full bg-success-light text-success dark:bg-success dark:text-success-light">
                                                    <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <path d="M4.72848 16.1369C3.18295 14.5914 2.41018 13.8186 2.12264 12.816C1.83509 11.8134 2.08083 10.7485 2.57231 8.61875L2.85574 7.39057C3.26922 5.59881 3.47597 4.70292 4.08944 4.08944C4.70292 3.47597 5.59881 3.26922 7.39057 2.85574L8.61875 2.57231C10.7485 2.08083 11.8134 1.83509 12.816 2.12264C13.8186 2.41018 14.5914 3.18295 16.1369 4.72848L17.9665 6.55812C20.6555 9.24711 22 10.5916 22 12.2623C22 13.933 20.6555 15.2775 17.9665 17.9665C15.2775 20.6555 13.933 22 12.2623 22C10.5916 22 9.24711 20.6555 6.55812 17.9665L4.72848 16.1369Z" stroke="currentColor" stroke-width="1.5"></path>
                                                        <circle opacity="0.5" cx="8.60699" cy="8.87891" r="2" transform="rotate(-45 8.60699 8.87891)" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <path opacity="0.5" d="M11.5417 18.5L18.5208 11.5208" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                            <div class="flex-1">
                                                <div class="mb-2 flex font-semibold text-white-dark">
                                                    <h6>Total Successful Transactions</h6>
                                                    <p class="ltr:ml-auto rtl:mr-auto">GH¢ <?= number_format($totalPaid, 2) ?></p>
                                                </div>
                                                <div class="h-2 w-full rounded-full bg-dark-light shadow dark:bg-[#1b2e4b]">
                                                    <div class="h-full w-full rounded-full bg-gradient-to-r from-[#3cba92] to-[#0ba360]" style="width: 65%"></div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="flex items-center">
                                            <div class="h-9 w-9 ltr:mr-3 rtl:ml-3">
                                                <div class="grid h-9 w-9 place-content-center rounded-full bg-warning-light text-warning dark:bg-warning dark:text-warning-light">
                                                    <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <path d="M2 12C2 8.22876 2 6.34315 3.17157 5.17157C4.34315 4 6.22876 4 10 4H14C17.7712 4 19.6569 4 20.8284 5.17157C22 6.34315 22 8.22876 22 12C22 15.7712 22 17.6569 20.8284 18.8284C19.6569 20 17.7712 20 14 20H10C6.22876 20 4.34315 20 3.17157 18.8284C2 17.6569 2 15.7712 2 12Z" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path opacity="0.5" d="M10 16H6" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                        <path opacity="0.5" d="M14 16H12.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                        <path opacity="0.5" d="M2 10L22 10" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                            <div class="flex-1">
                                                <div class="mb-2 flex font-semibold text-white-dark">
                                                    <h6>Total Failed Transactions</h6>
                                                    <p class="ltr:ml-auto rtl:mr-auto">GH¢ <?= number_format($totalFailed, 2) ?></p>
                                                </div>
                                                <div class="h-2 w-full rounded-full bg-dark-light shadow dark:bg-[#1b2e4b]">
                                                    <div class="h-full w-full rounded-full bg-gradient-to-r from-[#f09819] to-[#ff5858]" style="width: 80%"></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="panel h-full p-0">
                                    <div class="absolute flex w-full items-center justify-between p-5">
                                        <div class="relative">
                                            <div class="flex h-11 w-11 items-center justify-center rounded-lg bg-success-light text-success dark:bg-success dark:text-success-light">
                                                <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M2 3L2.26491 3.0883C3.58495 3.52832 4.24497 3.74832 4.62248 4.2721C5 4.79587 5 5.49159 5 6.88304V9.5C5 12.3284 5 13.7426 5.87868 14.6213C6.75736 15.5 8.17157 15.5 11 15.5H19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <path opacity="0.5" d="M7.5 18C8.32843 18 9 18.6716 9 19.5C9 20.3284 8.32843 21 7.5 21C6.67157 21 6 20.3284 6 19.5C6 18.6716 6.67157 18 7.5 18Z" stroke="currentColor" stroke-width="1.5"></path>
                                                    <path opacity="0.5" d="M16.5 18.0001C17.3284 18.0001 18 18.6716 18 19.5001C18 20.3285 17.3284 21.0001 16.5 21.0001C15.6716 21.0001 15 20.3285 15 19.5001C15 18.6716 15.6716 18.0001 16.5 18.0001Z" stroke="currentColor" stroke-width="1.5"></path>
                                                    <path opacity="0.5" d="M11 9H8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <path d="M5 6H16.4504C18.5054 6 19.5328 6 19.9775 6.67426C20.4221 7.34853 20.0173 8.29294 19.2078 10.1818L18.7792 11.1818C18.4013 12.0636 18.2123 12.5045 17.8366 12.7523C17.4609 13 16.9812 13 16.0218 13H5" stroke="currentColor" stroke-width="1.5"></path>
                                                </svg>
                                            </div>
                                        </div>
                                        <h5 class="text-2xl font-semibold ltr:text-right rtl:text-left dark:text-white-light">
                                            <?= number_format($total_orders_this_month) ?>
                                            <span class="block text-sm font-normal">Total Orders</span>
                                        </h5>
                                    </div>
                                    <div x-ref="totalOrders" class="overflow-hidden rounded-lg bg-transparent">
                                        <!-- loader -->
                                        <div class="grid min-h-[290px] place-content-center bg-white-light/30 dark:bg-dark dark:bg-opacity-[0.08]">
                                            <span class="inline-flex h-5 w-5 animate-spin rounded-full border-2 border-black !border-l-transparent dark:border-white"></span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-6 grid gap-6 sm:grid-cols-2 xl:grid-cols-3">
                                <!-- <div class="panel h-full pb-0 sm:col-span-2 xl:col-span-1">
                                    <h5 class="mb-5 text-lg font-semibold dark:text-white-light">Recent Activities  N/A</h5>

                                    <div class="perfect-scrollbar relative -mr-3 mb-4 h-[290px] pr-3">
                                        <div class="cursor-pointer text-sm">
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-primary ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Updated Server Logs</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">Just Now</div>

                                                <span class="badge badge-outline-primary absolute bg-primary-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-success ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Send Mail to HR and Admin</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">2 min ago</div>

                                                <span class="badge badge-outline-success absolute bg-success-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-danger ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Backup Files EOD</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">14:00</div>

                                                <span class="badge badge-outline-danger absolute bg-danger-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-black ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Collect documents from Sara</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">16:00</div>

                                                <span class="badge badge-outline-dark absolute bg-dark-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-warning ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Conference call with Marketing Manager.</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">17:00</div>

                                                <span class="badge badge-outline-warning absolute bg-warning-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">In progress</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-info ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Rebooted Server</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">17:00</div>

                                                <span class="badge badge-outline-info absolute bg-info-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-secondary ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Send contract details to Freelancer</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">18:00</div>

                                                <span class="badge badge-outline-secondary absolute bg-secondary-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-primary ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Updated Server Logs</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">Just Now</div>

                                                <span class="badge badge-outline-primary absolute bg-primary-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-success ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Send Mail to HR and Admin</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">2 min ago</div>

                                                <span class="badge badge-outline-success absolute bg-success-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-danger ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Backup Files EOD</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">14:00</div>

                                                <span class="badge badge-outline-danger absolute bg-danger-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-black ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Collect documents from Sara</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">16:00</div>

                                                <span class="badge badge-outline-dark absolute bg-dark-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-warning ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Conference call with Marketing Manager.</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">17:00</div>

                                                <span class="badge badge-outline-warning absolute bg-warning-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">In progress</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-info ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Rebooted Server</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">17:00</div>

                                                <span class="badge badge-outline-info absolute bg-info-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Completed</span>
                                            </div>
                                            <div class="group relative flex items-center py-1.5">
                                                <div class="h-1.5 w-1.5 rounded-full bg-secondary ltr:mr-1 rtl:ml-1.5"></div>
                                                <div class="flex-1">Send contract details to Freelancer</div>
                                                <div class="text-xs text-white-dark ltr:ml-auto rtl:mr-auto dark:text-gray-500">18:00</div>

                                                <span class="badge badge-outline-secondary absolute bg-secondary-light text-xs opacity-0 group-hover:opacity-100 ltr:right-0 rtl:left-0 dark:bg-[#0e1726]">Pending</span>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="border-t border-white-light dark:border-white/10">
                                        <a href="javascript:;" class="group group flex items-center justify-center p-4 font-semibold hover:text-primary">
                                            View All
                                            <svg class="h-4 w-4 transition duration-300 group-hover:translate-x-1 ltr:ml-1 rtl:mr-1 rtl:rotate-180 rtl:group-hover:-translate-x-1" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M4 12H20M20 12L14 6M20 12L14 18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </div> -->

                                <!-- <div class="panel h-full">
                                    <div class="mb-5 flex items-center justify-between dark:text-white-light">
                                        <h5 class="text-lg font-semibold">Transactions  N/A</h5>
                                        <div x-data="dropdown" @click.outside="open = false" class="dropdown">
                                            <a href="javascript:;" @click="toggle">
                                                <svg class="h-5 w-5 text-black/70 hover:!text-primary dark:text-white/70" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="5" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle opacity="0.5" cx="12" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <circle cx="19" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                </svg>
                                            </a>
                                            <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="ltr:right-0 rtl:left-0">
                                                <li><a href="javascript:;" @click="toggle">View Report</a></li>
                                                <li><a href="javascript:;" @click="toggle">Edit Report</a></li>
                                                <li><a href="javascript:;" @click="toggle">Mark as Done</a></li>
                                            </ul>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="space-y-6">
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-success-light text-base text-success dark:bg-success dark:text-success-light">SP</span>
                                                <div class="flex-1 px-3">
                                                    <div>Pluto Wifi</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">10 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-success ltr:ml-auto rtl:mr-auto">+$36.11</span>
                                            </div>
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-warning-light text-warning dark:bg-warning dark:text-warning-light">
                                                    <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="h-6 w-6">
                                                        <path d="M2 10C2 7.17157 2 5.75736 2.87868 4.87868C3.75736 4 5.17157 4 8 4H13C15.8284 4 17.2426 4 18.1213 4.87868C19 5.75736 19 7.17157 19 10C19 12.8284 19 14.2426 18.1213 15.1213C17.2426 16 15.8284 16 13 16H8C5.17157 16 3.75736 16 2.87868 15.1213C2 14.2426 2 12.8284 2 10Z" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path opacity="0.5" d="M19.0003 7.07617C19.9754 7.17208 20.6317 7.38885 21.1216 7.87873C22.0003 8.75741 22.0003 10.1716 22.0003 13.0001C22.0003 15.8285 22.0003 17.2427 21.1216 18.1214C20.2429 19.0001 18.8287 19.0001 16.0003 19.0001H11.0003C8.17187 19.0001 6.75766 19.0001 5.87898 18.1214C5.38909 17.6315 5.17233 16.9751 5.07642 16" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path d="M13 10C13 11.3807 11.8807 12.5 10.5 12.5C9.11929 12.5 8 11.3807 8 10C8 8.61929 9.11929 7.5 10.5 7.5C11.8807 7.5 13 8.61929 13 10Z" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path opacity="0.5" d="M16 12L16 8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                        <path opacity="0.5" d="M5 12L5 8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    </svg>
                                                </span>
                                                <div class="flex-1 px-3">
                                                    <div>Cash withdrawal</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">04 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-danger ltr:ml-auto rtl:mr-auto">-$16.44</span>
                                            </div>
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-danger-light text-danger dark:bg-danger dark:text-danger-light">
                                                    <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <circle cx="12" cy="6" r="4" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <path opacity="0.5" d="M20 17.5C20 19.9853 20 22 12 22C4 22 4 19.9853 4 17.5C4 15.0147 7.58172 13 12 13C16.4183 13 20 15.0147 20 17.5Z" stroke="currentColor" stroke-width="1.5"></path>
                                                    </svg>
                                                </span>
                                                <div class="flex-1 px-3">
                                                    <div>Amy Diaz</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">10 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-success ltr:ml-auto rtl:mr-auto">+$66.44</span>
                                            </div>
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-secondary-light text-secondary dark:bg-secondary dark:text-secondary-light">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 24 24">
                                                        <path fill="currentColor" d="M5.398 0v.006c3.028 8.556 5.37 15.175 8.348 23.596c2.344.058 4.85.398 4.854.398c-2.8-7.924-5.923-16.747-8.487-24zm8.489 0v9.63L18.6 22.951c-.043-7.86-.004-15.913.002-22.95zM5.398 1.05V24c1.873-.225 2.81-.312 4.715-.398v-9.22z"></path>
                                                    </svg>
                                                </span>
                                                <div class="flex-1 px-3">
                                                    <div>Netflix</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">04 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-danger ltr:ml-auto rtl:mr-auto">-$32.00</span>
                                            </div>
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-info-light text-base text-info dark:bg-info dark:text-info-light">DA</span>
                                                <div class="flex-1 px-3">
                                                    <div>Daisy Anderson</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">10 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-success ltr:ml-auto rtl:mr-auto">+$10.08</span>
                                            </div>
                                            <div class="flex">
                                                <span class="grid h-9 w-9 shrink-0 place-content-center rounded-md bg-primary-light text-primary dark:bg-primary dark:text-primary-light">
                                                    <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <path d="M13.926 9.70541C13.5474 9.33386 13.5474 8.74151 13.5474 7.55682V7.24712C13.5474 3.96249 13.5474 2.32018 12.6241 2.03721C11.7007 1.75425 10.711 3.09327 8.73167 5.77133L5.66953 9.91436C4.3848 11.6526 3.74244 12.5217 4.09639 13.205C4.10225 13.2164 4.10829 13.2276 4.1145 13.2387C4.48945 13.9117 5.59888 13.9117 7.81775 13.9117C9.05079 13.9117 9.6673 13.9117 10.054 14.2754" stroke="currentColor" stroke-width="1.5"></path>
                                                        <path opacity="0.5" d="M13.9259 9.70557L13.9459 9.72481C14.3326 10.0885 14.9492 10.0885 16.1822 10.0885C18.4011 10.0885 19.5105 10.0885 19.8854 10.7615C19.8917 10.7726 19.8977 10.7838 19.9036 10.7951C20.2575 11.4785 19.6151 12.3476 18.3304 14.0858L15.2682 18.2288C13.2888 20.9069 12.2991 22.2459 11.3758 21.9629C10.4524 21.68 10.4524 20.0376 10.4525 16.753L10.4525 16.4434C10.4525 15.2587 10.4525 14.6663 10.074 14.2948L10.054 14.2755" stroke="currentColor" stroke-width="1.5"></path>
                                                    </svg>
                                                </span>
                                                <div class="flex-1 px-3">
                                                    <div>Electricity Bill</div>
                                                    <div class="text-xs text-white-dark dark:text-gray-500">04 Jan 1:00PM</div>
                                                </div>
                                                <span class="whitespace-pre px-1 text-base text-danger ltr:ml-auto rtl:mr-auto">-$22.00</span>
                                            </div>
                                        </div>
                                    </div>
                                </div> -->

                                <!-- <div class="panel h-full overflow-hidden border-0 p-0">
                                    <div class="min-h-[190px] bg-gradient-to-r from-[#4361ee] to-[#160f6b] p-6">
                                        <div class="mb-6 flex items-center justify-between">
                                            <div class="flex items-center rounded-full bg-black/50 p-1 font-semibold text-white ltr:pr-3 rtl:pl-3">
                                                <img class="block h-8 w-8 rounded-full border-2 border-white/50 object-cover ltr:mr-1 rtl:ml-1" src="assets/images/profile-34.jpeg" alt="image">
                                                Pluto Wifi
                                            </div>
                                            <button type="button" class="flex h-9 w-9 items-center justify-between rounded-md bg-black text-white hover:opacity-80 ltr:ml-auto rtl:mr-auto">
                                                <svg class="m-auto h-6 w-6" viewbox="0 0 24 24" stroke="currentColor" stroke-width="1.5" fill="none" stroke-linecap="round" stroke-linejoin="round">
                                                    <line x1="12" y1="5" x2="12" y2="19"></line>
                                                    <line x1="5" y1="12" x2="19" y2="12"></line>
                                                </svg>
                                            </button>
                                        </div>
                                        <div class="flex items-center justify-between text-white">
                                            <p class="text-xl">Wallet Balance  N/A</p>
                                            <h5 class="text-2xl ltr:ml-auto rtl:mr-auto"><span class="text-white-light">GH¢</span>2,953.00</h5>
                                        </div>
                                    </div>
                                    <div class="-mt-12 grid grid-cols-2 gap-2 px-8">
                                        <div class="rounded-md bg-white px-4 py-2.5 shadow dark:bg-[#060818]">
                                            <span class="mb-4 flex items-center justify-between dark:text-white">Received
                                                <svg class="h-4 w-4 text-success" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M19 15L12 9L5 15" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </span>
                                            <div class="btn w-full border-0 bg-[#ebedf2] py-1 text-base text-[#515365] shadow-none dark:bg-black dark:text-[#bfc9d4]">
                                                GH¢ 97.99
                                            </div>
                                        </div>
                                        <div class="rounded-md bg-white px-4 py-2.5 shadow dark:bg-[#060818]">
                                            <span class="mb-4 flex items-center justify-between dark:text-white">Spent
                                                <svg class="h-4 w-4 text-danger" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M19 9L12 15L5 9" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </span>
                                            <div class="btn w-full border-0 bg-[#ebedf2] py-1 text-base text-[#515365] shadow-none dark:bg-black dark:text-[#bfc9d4]">
                                                GH¢ 53.00
                                            </div>
                                        </div>
                                    </div>
                                    <div class="p-5">
                                        <div class="mb-5">
                                            <span class="rounded-full bg-[#1b2e4b] px-4 py-1.5 text-xs text-white before:inline-block before:h-1.5 before:w-1.5 before:rounded-full before:bg-white ltr:before:mr-2 rtl:before:ml-2">Pending</span>
                                        </div>
                                        <div class="mb-5 space-y-1">
                                            <div class="flex items-center justify-between">
                                                <p class="font-semibold text-[#515365]">Netflix</p>
                                                <p class="text-base"><span>GH¢</span> <span class="font-semibold">13.85</span></p>
                                            </div>
                                            <div class="flex items-center justify-between">
                                                <p class="font-semibold text-[#515365]">BlueHost VPN</p>
                                                <p class="text-base"><span>GH¢</span> <span class="font-semibold">15.66</span></p>
                                            </div>
                                        </div>
                                        <div class="flex justify-around px-2 text-center">
                                            <button type="button" class="btn btn-secondary ltr:mr-2 rtl:ml-2">View Details</button>
                                            <button type="button" class="btn btn-success">Pay Now GH¢ 29.51</button>
                                        </div>
                                    </div>
                                </div> -->
                            </div>

                            <div class="grid grid-cols-1 gap-6 lg:grid-cols-2">
                                <div class="panel h-full w-full">
                                    <div class="mb-5 flex items-center justify-between">
                                        <h5 class="text-lg font-semibold dark:text-white-light">Recent Orders</h5>
                                    </div>
                                    <div class="table-responsive">
                                        <table>
                                            <thead>
                                                <tr>
                                                    <th class="ltr:rounded-l-md rtl:rounded-r-md">Username</th>
                                                    <th>Profile</th>
                                                    <th>Transaction ID</th>
                                                    <th>Price</th>
                                                    <th class="ltr:rounded-r-md rtl:rounded-l-md">Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php
                                                    $sql = "SELECT username, profile, id, amount, status FROM transaction_history WHERE status != ''";
                                                    if ($userWhere) {
                                                        $sql .= $userWhere;
                                                    }
                                                    $sql .= " ORDER BY created_at DESC LIMIT 5";
                                                    $stmt = $conn->prepare($sql);
                                                    if ($userWhere) {
                                                        $param1 = $userParams[0];
                                                        $param2 = $userParams[1];
                                                        $stmt->bind_param("ss", $param1, $param2);
                                                    }
                                                    $stmt->execute();
                                                    $result = $stmt->get_result();
                                                    if ($result && $result->num_rows > 0):
                                                        while ($row = $result->fetch_assoc()):
                                                            // Status badge
                                                            $status = ucfirst($row['status']);
                                                            $badgeClass = 'badge bg-info shadow-md';
                                                            if (strtolower($status) === 'paid') $badgeClass = 'badge bg-success shadow-md';
                                                            elseif (strtolower($status) === 'shipped') $badgeClass = 'badge bg-secondary shadow-md';
                                                            elseif (strtolower($status) === 'pending') $badgeClass = 'badge bg-warning shadow-md';
                                                    ?>
                                                <tr class="group text-white-dark hover:text-black dark:hover:text-white-light/90">
                                                    <td class="min-w-[150px] text-black dark:text-white">
                                                        <div class="flex items-center">
                                                            <img class="h-8 w-8 rounded-md object-cover ltr:mr-3 rtl:ml-3" src="assets/images/profile-6.jpeg" alt="avatar">
                                                            <span class="whitespace-nowrap"><?= htmlspecialchars($row['username']) ?></span>
                                                        </div>
                                                    </td>
                                                    <td class="text-primary"><?= htmlspecialchars($row['profile']) ?></td>
                                                    <td><a href="apps-invoice-preview.php?id=<?= $row['id'] ?>">#<?= $row['id'] ?></a></td>
                                                    <td>GH¢<?= number_format($row['amount'], 2) ?></td>
                                                    <td><span class="<?= $badgeClass ?>"><?= $status ?></span></td>
                                                </tr>
                                                <?php endwhile; else: ?>
                                                <tr>
                                                    <td colspan="5" class="text-center">No recent orders found.</td>
                                                </tr>
                                                <?php endif; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>

                                <div class="panel h-full w-full">
                                    <div class="mb-5 flex items-center justify-between">
                                        <h5 class="text-lg font-semibold dark:text-white-light">Top 5 Selling Products This Week</h5>
                                    </div>
                                    <div class="table-responsive">
                                        <table>
                                            <thead>
                                                <tr class="border-b-0">
                                                    <th class="ltr:rounded-l-md rtl:rounded-r-md">Profile</th>
                                                    <th>Sold</th>
                                                    <th>Unit Amount</th>
                                                    <th>Total Amount</th>
                                                    <th class="ltr:rounded-r-md rtl:rounded-l-md">Source</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($profileStats as $row): ?>
                                                    <tr class="group text-white-dark hover:text-black dark:hover:text-white-light/90">
                                                        <td class="min-w-[150px] text-black dark:text-white">
                                                            <div class="flex">
                                                                <img class="h-8 w-8 rounded-md object-cover ltr:mr-3 rtl:ml-3" src="assets/images/product-headphones.jpg" alt="avatar">
                                                                <p class="whitespace-nowrap">
                                                                    <?= htmlspecialchars($row['profile']) ?>
                                                                    <span class="block text-xs text-primary"><?= htmlspecialchars($row['source'] ?? 'Direct') ?></span>
                                                                </p>
                                                            </div>
                                                        </td>
                                                        <td><?= number_format($row['sold']) ?></td>
                                                        <td>GH¢<?= number_format($row['unit_amount'], 2) ?></td>
                                                        <td>GH¢<?= number_format($row['total_amount'], 2) ?></td>
                                                        <td>
                                                            <a class="flex items-center text-danger" href="javascript:;">
                                                                <svg class="h-3.5 w-3.5 ltr:mr-1 rtl:ml-1 rtl:rotate-180" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                    <path d="M12.6644 5.47875L16.6367 9.00968C18.2053 10.404 18.9896 11.1012 18.9896 11.9993C18.9896 12.8975 18.2053 13.5946 16.6367 14.989L12.6644 18.5199C11.9484 19.1563 11.5903 19.4746 11.2952 19.342C11 19.2095 11 18.7305 11 17.7725V15.4279C7.4 15.4279 3.5 17.1422 2 19.9993C2 10.8565 7.33333 8.57075 11 8.57075V6.22616C11 5.26817 11 4.78917 11.2952 4.65662C11.5903 4.52407 11.9484 4.8423 12.6644 5.47875Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                                    <path opacity="0.5" d="M15.5386 4.5L20.7548 9.34362C21.5489 10.081 22.0001 11.1158 22.0001 12.1994C22.0001 13.3418 21.4989 14.4266 20.629 15.1671L15.5386 19.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                                </svg>
                                                                <?= htmlspecialchars($row['source'] ?? 'Direct') ?>
                                                            </a>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- end main content section -->
                </div>

                <!-- start footer section -->
                <?php include 'footer.php'; ?>
                <!-- end footer section -->
            </div>
        </div>

        <script src="assets/js/alpine-collaspe.min.js"></script>
        <script src="assets/js/alpine-persist.min.js"></script>
        <script defer="" src="assets/js/alpine-ui.min.js"></script>
        <script defer="" src="assets/js/alpine-focus.min.js"></script>
        <script defer="" src="assets/js/alpine.min.js"></script>
        <script src="assets/js/custom.js"></script>
        <script defer="" src="assets/js/apexcharts.js"></script>

        <script>
            document.addEventListener('alpine:init', () => {
                // main section
                Alpine.data('scrollToTop', () => ({
                    showTopButton: false,
                    init() {
                        window.onscroll = () => {
                            this.scrollFunction();
                        };
                    },

                    scrollFunction() {
                        if (document.body.scrollTop > 50 || document.documentElement.scrollTop > 50) {
                            this.showTopButton = true;
                        } else {
                            this.showTopButton = false;
                        }
                    },

                    goToTop() {
                        document.body.scrollTop = 0;
                        document.documentElement.scrollTop = 0;
                    },
                }));

                // theme customization
                Alpine.data('customizer', () => ({
                    showCustomizer: false,
                }));

                // sidebar section
                Alpine.data('sidebar', () => ({
                    init() {
                        const selector = document.querySelector('.sidebar ul a[href="' + window.location.pathname + '"]');
                        if (selector) {
                            selector.classList.add('active');
                            const ul = selector.closest('ul.sub-menu');
                            if (ul) {
                                let ele = ul.closest('li.menu').querySelectorAll('.nav-link');
                                if (ele) {
                                    ele = ele[0];
                                    setTimeout(() => {
                                        ele.click();
                                    });
                                }
                            }
                        }
                    },
                }));

                // header section
                Alpine.data('header', () => ({
                    init() {
                        const selector = document.querySelector('ul.horizontal-menu a[href="' + window.location.pathname + '"]');
                        if (selector) {
                            selector.classList.add('active');
                            const ul = selector.closest('ul.sub-menu');
                            if (ul) {
                                let ele = ul.closest('li.menu').querySelectorAll('.nav-link');
                                if (ele) {
                                    ele = ele[0];
                                    setTimeout(() => {
                                        ele.classList.add('active');
                                    });
                                }
                            }
                        }
                    },

                    notifications: [
                        {
                            id: 1,
                            profile: 'user-profile.jpeg',
                            message: '<strong class="text-sm mr-1">Pluto Wifi</strong>invite you to <strong>Prototyping</strong>',
                            time: '45 min ago',
                        },
                        {
                            id: 2,
                            profile: 'profile-34.jpeg',
                            message: '<strong class="text-sm mr-1">Adam Nolan</strong>mentioned you to <strong>UX Basics</strong>',
                            time: '9h Ago',
                        },
                        {
                            id: 3,
                            profile: 'profile-16.jpeg',
                            message: '<strong class="text-sm mr-1">Anna Morgan</strong>Upload a file',
                            time: '9h Ago',
                        },
                    ],

                    messages: [
                        {
                            id: 1,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-success-light dark:bg-success text-success dark:text-success-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path></svg></span>',
                            title: 'Congratulations!',
                            message: 'Your OS has been updated.',
                            time: '1hr',
                        },
                        {
                            id: 2,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-info-light dark:bg-info text-info dark:text-info-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="12" y1="16" x2="12" y2="12"></line><line x1="12" y1="8" x2="12.01" y2="8"></line></svg></span>',
                            title: 'Did you know?',
                            message: 'You can switch between artboards.',
                            time: '2hr',
                        },
                        {
                            id: 3,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-danger-light dark:bg-danger text-danger dark:text-danger-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg></span>',
                            title: 'Something went wrong!',
                            message: 'Send Reposrt',
                            time: '2days',
                        },
                        {
                            id: 4,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-warning-light dark:bg-warning text-warning dark:text-warning-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">    <circle cx="12" cy="12" r="10"></circle>    <line x1="12" y1="8" x2="12" y2="12"></line>    <line x1="12" y1="16" x2="12.01" y2="16"></line></svg></span>',
                            title: 'Warning',
                            message: 'Your password strength is low.',
                            time: '5days',
                        },
                    ],

                    languages: [
                        {
                            id: 1,
                            key: 'Khmer',
                            value: 'kh',
                        },
                        {
                            id: 2,
                            key: 'Danish',
                            value: 'da',
                        },
                        {
                            id: 3,
                            key: 'English',
                            value: 'en',
                        },
                        {
                            id: 4,
                            key: 'French',
                            value: 'fr',
                        },
                        {
                            id: 5,
                            key: 'German',
                            value: 'de',
                        },
                        {
                            id: 6,
                            key: 'Greek',
                            value: 'el',
                        },
                        {
                            id: 7,
                            key: 'Hungarian',
                            value: 'hu',
                        },
                        {
                            id: 8,
                            key: 'Italian',
                            value: 'it',
                        },
                        {
                            id: 9,
                            key: 'Japanese',
                            value: 'ja',
                        },
                        {
                            id: 10,
                            key: 'Polish',
                            value: 'pl',
                        },
                        {
                            id: 11,
                            key: 'Portuguese',
                            value: 'pt',
                        },
                        {
                            id: 12,
                            key: 'Russian',
                            value: 'ru',
                        },
                        {
                            id: 13,
                            key: 'Spanish',
                            value: 'es',
                        },
                        {
                            id: 14,
                            key: 'Swedish',
                            value: 'sv',
                        },
                        {
                            id: 15,
                            key: 'Turkish',
                            value: 'tr',
                        },
                        {
                            id: 16,
                            key: 'Arabic',
                            value: 'ae',
                        },
                    ],

                    removeNotification(value) {
                        this.notifications = this.notifications.filter((d) => d.id !== value);
                    },

                    removeMessage(value) {
                        this.messages = this.messages.filter((d) => d.id !== value);
                    },
                }));

                // content section
                Alpine.data('sales', () => ({
                    mode: 'weekly',
                    init() {
                        isDark = this.$store.app.theme === 'dark' || this.$store.app.isDarkMode ? true : false;
                        isRtl = this.$store.app.rtlClass === 'rtl' ? true : false;

                        const revenueChart = null;
                        const salesByCategory = null;
                        const dailySales = null;
                        const totalOrders = null;

                        // revenue
                        setTimeout(() => {
                            this.revenueChart = new ApexCharts(this.$refs.revenueChart, this.revenueChartOptions);
                            this.$refs.revenueChart.innerHTML = '';
                            this.revenueChart.render();

                            // sales by category
                            this.salesByCategory = new ApexCharts(this.$refs.salesByCategory, this.salesByCategoryOptions);
                            this.$refs.salesByCategory.innerHTML = '';
                            this.salesByCategory.render();

                            // daily sales
                            this.dailySales = new ApexCharts(this.$refs.dailySales, this.dailySalesOptions);
                            this.$refs.dailySales.innerHTML = '';
                            this.dailySales.render();

                            // total orders
                            this.totalOrders = new ApexCharts(this.$refs.totalOrders, this.totalOrdersOptions);
                            this.$refs.totalOrders.innerHTML = '';
                            this.totalOrders.render();
                        }, 300);

                        this.$watch('$store.app.theme', () => {
                            isDark = this.$store.app.theme === 'dark' || this.$store.app.isDarkMode ? true : false;

                            this.revenueChart.updateOptions(this.revenueChartOptions);
                            this.salesByCategory.updateOptions(this.salesByCategoryOptions);
                            this.dailySales.updateOptions(this.dailySalesOptions);
                            this.totalOrders.updateOptions(this.totalOrdersOptions);
                        });

                        this.$watch('$store.app.rtlClass', () => {
                            isRtl = this.$store.app.rtlClass === 'rtl' ? true : false;
                            this.revenueChart.updateOptions(this.revenueChartOptions);
                        });
                    },

                    // revenue
                    get revenueChartOptions() {
                        let mode = this.mode || 'weekly';
                        if (mode === 'weekly') {
                            return {
                                series: [
                                    { name: 'This Week', data: <?= json_encode($revenue_this_week) ?> },
                                    { name: 'Last Week', data: <?= json_encode($revenue_last_week) ?> }
                                ],
                                chart: { height: 325, type: 'area', fontFamily: 'Nunito, sans-serif', zoom: { enabled: false }, toolbar: { show: false } },
                                xaxis: { categories: ['Mon','Tue','Wed','Thu','Fri','Sat','Sun'] },
                                // ...rest of your options...
                            };
                        } else if (mode === 'monthly') {
                            return {
                                series: [
                                    { name: 'This Month', data: <?= json_encode($revenue_this_month) ?> },
                                    { name: 'Last Month', data: <?= json_encode($revenue_last_month) ?> }
                                ],
                                chart: { height: 325, type: 'area', fontFamily: 'Nunito, sans-serif', zoom: { enabled: false }, toolbar: { show: false } },
                                xaxis: { categories: ['Week 1','Week 2','Week 3','Week 4','Week 5'] },
                                // ...rest of your options...
                            };
                        } else {
                            return {
                                series: [
                                    { name: 'This Year', data: <?= json_encode($revenue_this_year) ?> },
                                    { name: 'Last Year', data: <?= json_encode($revenue_last_year) ?> }
                                ],
                                chart: { height: 325, type: 'area', fontFamily: 'Nunito, sans-serif', zoom: { enabled: false }, toolbar: { show: false } },
                                xaxis: { categories: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'] },
                                // ...rest of your options...
                            };
                        }
                    },

                    setMode(mode) {
                        this.mode = mode;
                        setTimeout(() => {
                            this.revenueChart.updateOptions(this.revenueChartOptions);
                        }, 10);
                    },

                    // sales by category
                    get salesByCategoryOptions() {
                        return {
                            series: <?= json_encode($donutCounts) ?>,
                            chart: {
                                type: 'donut',
                                height: 460,
                                fontFamily: 'Nunito, sans-serif',
                            },
                            dataLabels: { enabled: false },
                            stroke: {
                                show: true,
                                width: 25,
                                colors: (this.$store.app.theme === 'dark' || this.$store.app.isDarkMode) ? '#0e1726' : '#fff',
                            },
                            colors: <?= json_encode(array_slice($donutColors, 0, count($donutCounts))) ?>,
                            legend: {
                                position: 'bottom',
                                horizontalAlign: 'center',
                                fontSize: '14px',
                                markers: { width: 10, height: 10, offsetX: -2 },
                                height: 50,
                                offsetY: 20,
                            },
                            plotOptions: {
                                pie: {
                                    donut: {
                                        size: '65%',
                                        background: 'transparent',
                                        labels: {
                                            show: true,
                                            name: { show: true, fontSize: '18px', offsetY: -10 },
                                            value: {
                                                show: true,
                                                fontSize: '22px',
                                                color: (this.$store.app.theme === 'dark' || this.$store.app.isDarkMode) ? '#bfc9d4' : undefined,
                                                offsetY: 16,
                                                formatter: (val) => Number(val).toLocaleString(),
                                            },
                                            total: {
                                                show: true,
                                                label: 'Total',
                                                color: '#888ea8',
                                                fontSize: '29px',
                                                formatter: () => <?= $totalToday ?>,
                                            },
                                        },
                                    },
                                },
                            },
                            labels: <?= json_encode($donutLabels) ?>,
                            states: {
                                hover: { filter: { type: 'none', value: 0.15 } },
                                active: { filter: { type: 'none', value: 0.15 } },
                            },
                        };
                    },

                    // daily sales
                    get dailySalesOptions() {
                        return {
                            series: [
                                {
                                    name: 'This Week',
                                    data: <?= json_encode($daily_sales_this_week) ?>,
                                },
                                {
                                    name: 'Last Week',
                                    data: <?= json_encode($daily_sales_last_week) ?>,
                                },
                            ],
                            chart: {
                                height: 160,
                                type: 'bar',
                                fontFamily: 'Nunito, sans-serif',
                                toolbar: { show: false },
                                stacked: true,
                                stackType: '100%',
                            },
                            dataLabels: { enabled: false },
                            stroke: { show: true, width: 1 },
                            colors: ['#e2a03f', '#e0e6ed'],
                            responsive: [
                                {
                                    breakpoint: 480,
                                    options: {
                                        legend: {
                                            position: 'bottom',
                                            offsetX: -10,
                                            offsetY: 0,
                                        },
                                    },
                                },
                            ],
                            xaxis: {
                                labels: { show: false },
                                categories: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                            },
                            yaxis: { show: false },
                            fill: { opacity: 1 },
                            plotOptions: {
                                bar: {
                                    horizontal: false,
                                    columnWidth: '25%',
                                },
                            },
                            legend: { show: false },
                            grid: {
                                show: false,
                                xaxis: { lines: { show: false } },
                                padding: { top: 10, right: -20, bottom: -20, left: -20 },
                            },
                        };
                    },

                    // total orders
                    get totalOrdersOptions() {
                        return {
                            series: [
                                {
                                    name: 'Orders',
                                    data: <?= json_encode($weekly_orders_this_month) ?>,
                                },
                            ],
                            chart: {
                                height: 290,
                                type: 'area',
                                fontFamily: 'Nunito, sans-serif',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                curve: 'smooth',
                                width: 2,
                            },
                            colors: isDark ? ['#00ab55'] : ['#00ab55'],
                            labels: ['Week 1', 'Week 2', 'Week 3', 'Week 4', 'Week 5'],
                            yaxis: {
                                min: 0,
                                show: false,
                            },
                            grid: {
                                padding: {
                                    top: 125,
                                    right: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            fill: {
                                opacity: 1,
                                type: 'gradient',
                                gradient: {
                                    type: 'vertical',
                                    shadeIntensity: 1,
                                    inverseColors: !1,
                                    opacityFrom: 0.3,
                                    opacityTo: 0.05,
                                    stops: [100, 100],
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                            },
                        };
                    },
                }));
            });
        </script>
        <script>
document.addEventListener('alpine:init', () => {
    Alpine.store('sales', {
        mode: 'weekly',
        setMode(mode) {
            this.mode = mode;
            this.renderChart();
        },
        renderChart() {
            let options;
            if (this.mode === 'weekly') {
                options = {
                    series: [
                        { name: 'This Week', data: <?= json_encode($revenue_this_week) ?> },
                        { name: 'Last Week', data: <?= json_encode($revenue_last_week) ?> }
                    ],
                    chart: { height: 325, type: 'area' },
                    xaxis: { categories: ['Mon','Tue','Wed','Thu','Fri','Sat','Sun'] },
                    yaxis: { title: { text: 'Revenue' } },
                    // ...other options...
                };
                document.getElementById('revenue-totals').innerHTML =
                    'Total Revenue This Week: GH¢ <?= number_format($total_this_week,2) ?> | Last Week: GH¢ <?= number_format($total_last_week,2) ?>';
            } else if (this.mode === 'monthly') {
                options = {
                    series: [
                        { name: 'This Month', data: <?= json_encode($revenue_this_month) ?> },
                        { name: 'Last Month', data: <?= json_encode($revenue_last_month) ?> }
                    ],
                    chart: { height: 325, type: 'area' },
                    xaxis: { categories: ['Week 1','Week 2','Week 3','Week 4','Week 5'] },
                    yaxis: { title: { text: 'Revenue' } },
                };
                document.getElementById('revenue-totals').innerHTML =
                    'Total Revenue This Month: GH¢ <?= number_format($total_this_month,2) ?> | Last Month: GH¢ <?= number_format($total_last_month,2) ?>';
            } else {
                options = {
                    series: [
                        { name: 'This Year', data: <?= json_encode($revenue_this_year) ?> },
                        { name: 'Last Year', data: <?= json_encode($revenue_last_year) ?> }
                    ],
                    chart: { height: 325, type: 'area' },
                    xaxis: { categories: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'] },
                    yaxis: { title: { text: 'Revenue' } },
                };
                document.getElementById('revenue-totals').innerHTML =
                    'Total Revenue This Year: GH¢ <?= number_format($total_this_year,2) ?> | Last Year: GH¢ <?= number_format($total_last_year,2) ?>';
            }
            if (window.revenueChart) window.revenueChart.destroy();
            window.revenueChart = new ApexCharts(document.querySelector("[x-ref='revenueChart']"), options);
            window.revenueChart.render();
        }
    });

    // Render default chart on load
    Alpine.store('sales').renderChart();
});
</script>
    </body>
</html>
